"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DenyList = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("./constants");
const create_map_1 = require("./create-map");
const prune_1 = require("./prune");
/**
 * Manages the construct hub deny list.
 */
class DenyList extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        /**
         * The object key within the bucket with the deny list JSON file.
         */
        this.objectKey = 'deny-list.json';
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            versioned: true,
        });
        const directory = this.writeToFile(props.rules, this.objectKey);
        // upload the deny list to the bucket
        this.upload = new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            destinationBucket: this.bucket,
            prune: true,
            retainOnDelete: false,
            sources: [s3deploy.Source.asset(directory)],
        });
        this.prune = new prune_1.Prune(this, 'Prune', {
            packageDataBucket: props.packageDataBucket,
            packageDataKeyPrefix: props.packageDataKeyPrefix,
            monitoring: props.monitoring,
        });
        this.grantRead(this.prune.pruneHandler);
        // trigger prune when the deny list changes
        const pruneOnChange = (_a = props.pruneOnChange) !== null && _a !== void 0 ? _a : true;
        if (pruneOnChange) {
            this.prune.pruneHandler.addEventSource(new aws_lambda_event_sources_1.S3EventSource(this.bucket, {
                events: [s3.EventType.OBJECT_CREATED],
                filters: [{ prefix: this.objectKey, suffix: this.objectKey }],
            }));
            // add an explicit dep between upload and the bucket scope which can now
            // also include the bucket notification resource. otherwise, the first
            // upload will not trigger a prune
            this.upload.node.addDependency(this.bucket);
        }
        // trigger prune periodically (every 5 minutes) - just in case
        const prunePeriod = (_b = props.prunePeriod) !== null && _b !== void 0 ? _b : core_1.Duration.minutes(5);
        if (prunePeriod && prunePeriod.toSeconds() > 0) {
            new events.Rule(this, 'PeriodicPrune', {
                schedule: events.Schedule.rate(prunePeriod),
                targets: [new targets.LambdaFunction(this.prune.pruneHandler)],
            });
        }
    }
    /**
     * Grants an AWS Lambda function permissions to read the deny list, and adds
     * the relevant environment variables expected by the `DenyListClient`.
     */
    grantRead(handler) {
        handler.addEnvironment(constants_1.ENV_DENY_LIST_BUCKET_NAME, this.bucket.bucketName);
        handler.addEnvironment(constants_1.ENV_DENY_LIST_OBJECT_KEY, this.objectKey);
        this.bucket.grantRead(handler);
    }
    /**
     * Number of rules in the deny list.
     */
    metricDenyListRules(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "DenyListRuleCount" /* DENY_LIST_RULE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * Writes the deny list to a temporary file and returns a path to a directory
     * with the JSON file. The contents of the JSON file is a map where keys are
     * package names (and optionally, version) and the value is the deny list
     * entry. This makes it easier to query by the service.
     *
     * Also performs some validation to make sure there are no duplicate entries.
     *
     * @param list The deny list
     * @returns a path to a temporary directory that can be deployed to S3
     */
    writeToFile(list, fileName) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'deny-list-'));
        const filePath = path.join(tmpdir, fileName);
        const map = create_map_1.createDenyListMap(list);
        fs.writeFileSync(filePath, JSON.stringify(map, null, 2));
        return tmpdir;
    }
}
exports.DenyList = DenyList;
//# sourceMappingURL=data:application/json;base64,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