"""
CLI tool for creating a test document from an test directory or JSON
"""

import argparse
import os
import logging
import tempfile

import k3logging

from k3testdocumentation_generator import __version__, generate_json_from_file_system, generate_html_test_documentation_from_json, generate_pdf_test_documentation_from_json
import shutil


__author__ = 'Joachim Kestner <joachim.kestner@khoch3.de>'

logger = logging.getLogger(__name__)

def main():
    parser = argparse.ArgumentParser(description=__doc__+"\n\nAuthor: {}\nVersion: {}".format(__author__,__version__), formatter_class=argparse.RawDescriptionHelpFormatter)
    #parser.add_argument("-f", "--flag", action="store_true", help="Example argparse of a choice")
    parser.add_argument("-t", "--output_type", default="PDF", choices=["PDF", "JSON", "HTML"], help="The output format. Default is PDF")
    parser.add_argument("-o", "--output", help="Output file path. If not set a name will be generated by: basename(input) + output_type.lower()")
    parser.add_argument("input", help="Input to generate documentation from. Can either be a directory containing the specified structure or an appropriate JSON")
    
    k3logging.set_parser_log_arguments(parser)
    
    args = parser.parse_args()
    
    k3logging.eval_parser_log_arguments(args)
    
    with tempfile.TemporaryDirectory() as tmpdir:
    
        if os.path.isdir(args.input):
            tmpjsonfile = os.path.join(tmpdir, "t.json")
            generate_json_from_file_system(args.input, tmpjsonfile)
            logger.info(f"Generated json file '{tmpjsonfile}'")
            jsonFilePath = tmpjsonfile
        else:
            jsonFilePath = args.input
        
        if not args.output:
            if args.input.endswith("/"):
                outputFilePath = os.path.basename(args.input[:-1]) + "." + args.output_type.lower()
            else:
                outputFilePath = os.path.basename(args.input) + "." + args.output_type.lower()
        else:
            outputFilePath = args.output
        
        if args.output_type == "JSON":
            shutil.copy(jsonFilePath, outputFilePath)
        elif args.output_type == "HTML":
            generate_html_test_documentation_from_json(jsonFilePath, outputFilePath)
        elif args.output_type == "PDF":
            generate_pdf_test_documentation_from_json(jsonFilePath, outputFilePath)
        else:
            raise RuntimeError("Invalid output type")
        
        logger.info(f"Output written to '{outputFilePath}'")
