# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['guest_user',
 'guest_user.management',
 'guest_user.management.commands',
 'guest_user.migrations',
 'guest_user.templatetags']

package_data = \
{'': ['*'], 'guest_user': ['templates/guest_user/*']}

setup_kwargs = {
    'name': 'django-guest-user',
    'version': '0.1.2',
    'description': 'A Django app that lets visitors interact with your site without registration.',
    'long_description': '[![Code Lint](https://github.com/julianwachholz/django-guest-user/actions/workflows/lint.yml/badge.svg)](https://github.com/julianwachholz/django-guest-user/actions/workflows/lint.yml)\n[![Python Tests](https://github.com/julianwachholz/django-guest-user/actions/workflows/test.yml/badge.svg)](https://github.com/julianwachholz/django-guest-user/actions/workflows/test.yml)\n\n# django-guest-user\n\nA Django app that allows visitors to interact with your site as a guest user\nwithout requiring registration.\n\nLargely inspired by [django-lazysignup](https://github.com/danfairs/django-lazysignup) and rewritten for Django 3 and Python 3.6 and up.\n\n## Installation\n\nInstall the package with your favorite package manager from PyPI:\n\n```\npip install django-guest-user\n```\n\nAdd the app to your `INSTALLED_APPS` and `AUTHENTICATION_BACKENDS`:\n\n```python\nINSTALLED_APPS = [\n    ...\n    "guest_user",\n]\n\nAUTHENTICATION_BACKENDS = [\n    "django.contrib.auth.backends.ModelBackend",\n    "guest_user.backends.GuestBackend",\n]\n```\n\nAdd the patterns to your URL config:\n\n```python\nurlpatterns = [\n    ...\n    path("convert/", include("guest_user.urls")),\n]\n```\n\nDon\'t forget to run migrations:\n\n```\npython manage.py migrate\n```\n\n## How to use\n\nGuest users are not created for every unauthenticated request.\nInstead, use the `@allow_guest_user` decorator on a view to enable\nthat view to be accessed by a temporary guest user.\n\n```python\nfrom guest_user.decorators import allow_guest_user\n\n@allow_guest_user\ndef my_view(request):\n    # Will always be either a registered a guest user.\n    username = request.user.username\n    return HttpResponse(f"Hello, {username}!")\n```\n\n## API\n\n### `@guest_user.decorators.allow_guest_user`\n\nView decorator that will create a temporary guest user in the event\nthat the decorated view is accessed by an unauthenticated visitor.\n\nTakes no arguments.\n\n### `@guest_user.decorators.guest_user_required(redirect_field_name="next", login_url=None)`\n\nView decorator that redirects to a given URL if the accessing user is\nanonymous or already authenticated.\n\nArguments:\n\n- `redirect_field_name`: URL query parameter to use to link back in the case of a redirect to the login url. Defaults to `django.contrib.auth.REDIRECT_FIELD_NAME` ("next").\n- `login_url`: URL to redirect to if the user is not authenticated. Defaults to the `LOGIN_URL` setting.\n\n### `@guest_user.decorators.regular_user_required(redirect_field_name="next", login_url=None)`\n\nDecorator that will not allow guest users to access the view.\nWill redirect to the conversion page to allow a guest user to fully register.\n\nArguments:\n\n- `redirect_field_name`: URL query parameter to use to link back in the case of a redirect to the login url. Defaults to `django.contrib.auth.REDIRECT_FIELD_NAME` ("next").\n- `login_url`: URL to redirect to if the user is a guest. Defaults to `"guest_user_convert"`.\n\n### `guest_user.functions.get_guest_model()`\n\nThe guest user model is swappable. This function will return the currently configured model class.\n\n### `guest_user.functions.is_guest_user(user)`\n\nCheck wether the given user instance is a temporary guest.\n\n### `guest_user.signals.converted`\n\nSignal that is dispatched when a guest user is converted to a regular user.\n\n### Template tag `is_guest_user`\n\nA filter to use in templates to check if the user object is a guest.\n\n```\n{% load guest_user %}\n\n{% if user|is_guest_user %}\n  Hello guest.\n{% endif %}\n```\n\n## Settings\n\nVarious settings are provided to allow customization of the guest user behavior.\n\n### `GUEST_USER_ENABLED`\n\n`bool`. If `False`, the `@allow_guest_user` decorator will not create guest users.\nDefaults to `True`.\n\n### `GUEST_USER_MODEL`\n\n`str`. The swappable model identifier to use as the guest model.\nDefaults to `"guest_user.Guest"`.\n\n### `GUEST_USER_NAME_GENERATOR`\n\n`str`. Import path to a function that will generate a username for a guest user.\nDefaults to `"guest_user.functions.generate_uuid_username"`.\n\nIncluded with the package are two alternatives:\n\n`"guest_user.functions.generate_numbered_username"`: Will create a random four digit\nnumber prefixed by `GUEST_USER_NAME_PREFIX`.\n\n`"guest_user.functions.generate_friendly_username"`: Creates a friendly and easy to remember username by combining an adjective, noun and number. Requires `random_username` to be installed.\n\n### `GUEST_USER_NAME_PREFIX`\n\n`str`. A prefix to use with the `generate_numbered_username` generator.\nDefaults to `"Guest"`.\n\n### `GUEST_USER_MAX_AGE`\n\n`int`. Seconds to keep a guest user before expiring.\nDefaults to `settings.SESSION_COOKIE_AGE`.\n\n### `GUEST_USER_CONVERT_FORM`\n\n`str`. Import path for the guest conversion form.\nMust implement `get_credentials` to be passed to Django\'s `authenticate` function.\nDefaults to `"guest_user.forms.UserCreationForm"`.\n\n### `GUEST_USER_CONVERT_PREFILL_USERNAME`\n\n`bool`. Set the generated username as initial value on the conversion form.\nDefaults to `False`.\n\n### `GUEST_USER_CONVERT_URL`\n\n`str`. URL name for the convert view.\nDefaults to `"guest_user_convert"`.\n\n### `GUEST_USER_CONVERT_REDIRECT_URL`\n\n`str`. URL name to redirect to after conversion, unless a redirect parameter was provided.\nDefaults to `"guest_user_convert_success"`.\n\n### `GUEST_USER_BLOCKED_USER_AGENTS`\n\n`list[str]`. Web crawlers and other user agents to block from becoming guest users.\nThe list will be combined into a regular expression.\nDefault includes a number of well known bots and spiders.\n\n## Status\n\nThis project is currently untested. But thanks to [previous work](https://github.com/danfairs/django-lazysignup) it is largely functional.\n\nI decided to rewrite the project since the original project hasn\'t seen any\nlarger updates for a few years now and the code base was written a long time ago.\n',
    'author': 'Julian Wachholz',
    'author_email': 'julian@wachholz.ch',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/julianwachholz/django-guest-user',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6.2,<4.0.0',
}


setup(**setup_kwargs)
