"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const app = new cdk.App();
const env = {
    region: process.env.CDK_DEFAULT_REGION,
    account: process.env.CDK_DEFAULT_ACCOUNT,
};
const stack = new cdk.Stack(app, 'demo-stack2', { env });
const zoneName = 'svc.local';
const internalAlbRecordName = 'internal';
const externalAlbRecordName = 'external';
const internalALBEndpoint = `http://${internalAlbRecordName}.${zoneName}`;
// order service
const orderTask = new ecs.FargateTaskDefinition(stack, 'orderTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
orderTask.addContainer('order', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/OrderService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'order',
        versionNum: '1.0',
    },
});
// customer service
const customerTask = new ecs.FargateTaskDefinition(stack, 'customerTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
customerTask.addContainer('customer', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'customer',
        versionNum: '1.0',
    },
});
// product service
const productTask = new ecs.FargateTaskDefinition(stack, 'productTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
productTask.addContainer('product', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/golang/CommonService')),
    portMappings: [
        { containerPort: 8080 },
    ],
    environment: {
        PRODUCT_SVC_URL: `${internalALBEndpoint}:9090`,
        CUSTOMER_SVC_URL: `${internalALBEndpoint}:8080`,
        serviceName: 'product',
        versionNum: '1.0',
    },
});
// nginx service
const nginxTask = new ecs.FargateTaskDefinition(stack, 'nginxTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nginxTask.addContainer('nginx', {
    image: ecs.ContainerImage.fromRegistry('nginx:latest'),
    portMappings: [
        { containerPort: 80 },
    ],
});
// php service
const phpTask = new ecs.FargateTaskDefinition(stack, 'phpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
phpTask.addContainer('php', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nginx-php')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// laravel-nginx-php-fpm service
const laravelNginxPhpFpmTask = new ecs.FargateTaskDefinition(stack, 'laravelNginxPhpFpmTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelNginxPhpFpmTask.addContainer('laravel-nginx-php-fpm', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-nginx-php-fpm')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// laravel-bitnami service
const laravelBitnamiTask = new ecs.FargateTaskDefinition(stack, 'laravelBitnamiTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
laravelBitnamiTask.addContainer('laravel-bitnami', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/laravel-bitnami')),
    portMappings: [
        { containerPort: 3000 },
    ],
});
// NuxtJS service
const nuxtTask = new ecs.FargateTaskDefinition(stack, 'nuxtTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nuxtTask.addContainer('nuxt', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/nuxt')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// Node.js service
const nodeTask = new ecs.FargateTaskDefinition(stack, 'nodeTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
nodeTask.addContainer('node', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/node')),
    portMappings: [
        { containerPort: 80 },
    ],
});
// java spring boot service
const javaTask = new ecs.FargateTaskDefinition(stack, 'javaTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
javaTask.addContainer('java', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/java-spring-boot')),
    portMappings: [
        { containerPort: 8080 },
    ],
});
// apache-php service
const apachePHP = new ecs.FargateTaskDefinition(stack, 'apachephpTask', {
    cpu: 256,
    memoryLimitMiB: 512,
});
apachePHP.addContainer('apachephp', {
    image: ecs.ContainerImage.fromAsset(path.join(__dirname, '../services/apache-php')),
    portMappings: [
        { containerPort: 80 },
    ],
});
const certArn = stack.node.tryGetContext('ACM_CERT_ARN');
const cert = certArn ? acm.Certificate.fromCertificateArn(stack, 'Cert', certArn) : undefined;
const svc = new index_1.DualAlbFargateService(stack, 'Service', {
    spot: true,
    enableExecuteCommand: true,
    tasks: [
        // The order service with both external/internal access
        {
            task: orderTask,
            desiredCount: 2,
            internal: { port: 80 },
            external: cert ? { port: 443, certificate: [cert] } : { port: 80 },
            // customize the service autoscaling policy
            scalingPolicy: {
                maxCapacity: 20,
                requestPerTarget: 1000,
                targetCpuUtilization: 50,
            },
        },
        {
            // The customer service(internal only)
            task: customerTask,
            desiredCount: 1,
            internal: { port: 8080 },
            capacityProviderStrategy: [
                {
                    capacityProvider: 'FARGATE',
                    base: 1,
                    weight: 1,
                },
                {
                    capacityProvider: 'FARGATE_SPOT',
                    base: 0,
                    weight: 3,
                },
            ],
        },
        // The produce service(internal only)
        {
            task: productTask,
            desiredCount: 1,
            internal: { port: 9090 },
        },
        // The nginx service(external only)
        {
            task: nginxTask,
            desiredCount: 1,
            external: { port: 9091 },
        },
        // The nginx-php-fpm service(external/internal)
        {
            task: phpTask,
            desiredCount: 1,
            internal: { port: 9092 },
            external: { port: 9092 },
        },
        // The NuxtJS service(external/internal)
        {
            task: nuxtTask,
            desiredCount: 1,
            internal: { port: 9093 },
            external: { port: 9093 },
        },
        // The node service(external/internal)
        {
            task: nodeTask,
            desiredCount: 1,
            internal: { port: 9094 },
            external: { port: 9094 },
        },
        // The laravel-nginx-php-fpm service(external/internal)
        {
            task: laravelNginxPhpFpmTask,
            desiredCount: 1,
            internal: { port: 9095 },
            external: { port: 9095 },
        },
        // The laravel-bitnami service(external/internal)
        {
            task: laravelBitnamiTask,
            desiredCount: 1,
            internal: { port: 9096 },
            external: { port: 9096 },
        },
        // java spring boot service(external/internal)
        {
            task: javaTask,
            desiredCount: 1,
            internal: { port: 9097 },
            external: { port: 9097 },
            healthCheck: { path: '/hello-world' },
        },
        // apache-php(external/internal)
        {
            task: apachePHP,
            desiredCount: 1,
            internal: { port: 9098 },
            external: { port: 9098 },
        },
    ],
    route53Ops: {
        zoneName,
        externalAlbRecordName,
        internalAlbRecordName,
    },
});
// create a dummy sg
const dummySg = new ec2.SecurityGroup(stack, 'DummySG', {
    vpc: svc.vpc,
});
// allow all traffic from dummy sg to all the services
for (let i = 0; i < svc.service.length; i++) {
    svc.service[i].connections.allowFrom(dummySg, ec2.Port.allTraffic());
}
//# sourceMappingURL=data:application/json;base64,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