"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const semver_1 = require("semver");
const aws_lambda_shared_1 = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
async function handler({ key, description, widgetContext }) {
    var _a, _b;
    console.log(`Event: ${JSON.stringify({ key, description, widgetContext }, null, 2)}`);
    try {
        const bucketName = env_lambda_shared_1.requireEnv('BUCKET_NAME');
        let { Body, ContentEncoding, LastModified } = await aws_lambda_shared_1.s3().getObject({
            Bucket: bucketName,
            Key: key,
        }).promise();
        // If it was compressed, de-compress it now...
        if (ContentEncoding === 'gzip') {
            Body = zlib_1.gunzipSync(Buffer.from(Body));
        }
        const list = Array.from(JSON.parse(Body.toString('utf-8'))
            .reduce((map, entry) => {
            // Split on the @ that is not at the beginning of the string
            const [name, version] = entry.split(/(?!^)@/);
            if (!map.has(name)) {
                map.set(name, []);
            }
            map.get(name).push(version);
            return map;
        }, new Map())
            .entries())
            .sort(([l], [r]) => l.localeCompare(r));
        // Trying to ensure we don't cause the dashboard to hang due to large DOM.
        const maxCount = 100;
        const objectUrl = `${widgetContext.domain}/s3/object/${bucketName}?prefix=${key}`;
        return {
            markdown: [
                description,
                ...(list.length > maxCount
                    ? [
                        `Showing only the first ${maxCount} packages.`,
                        `The complete list can be obtained [from S3](${objectUrl}).`,
                        '',
                    ]
                    : []),
                'Id | Package Name | Count | Versions',
                '--:|--------------|-------|---------',
                ...list.slice(0, maxCount).map(([name, versions], index) => {
                    versions = semver_1.sort(versions).reverse();
                    return `${index + 1} | \`${name}\` | ${versions.length} | ${versions.map((v) => `[\`${v}\`](${s3ConsoleUrl(bucketName, name, v)})`).join(', ')}`;
                }),
                '',
                `Last updated: \`${(_a = LastModified === null || LastModified === void 0 ? void 0 : LastModified.toISOString()) !== null && _a !== void 0 ? _a : 'N/A'}\``,
            ].join('\n'),
        };
    }
    catch (error) {
        if (error instanceof Error) {
            return {
                markdown: [
                    '**⚠️ An error occurred**',
                    `- **name:** \`${error.name}\``,
                    `- **message:** ${error.message}`,
                    '- **stack:**',
                    '  ```',
                    (_b = error.stack) === null || _b === void 0 ? void 0 : _b.replace(/^/g, '  '),
                    '  ```',
                ].join('\n'),
            };
        }
        ;
        throw error;
    }
}
exports.handler = handler;
function s3ConsoleUrl(bucket, packageName, packageVersion) {
    const encodedPrefix = encodeURIComponent(`data/${packageName}/v${packageVersion}/`);
    return `https://s3.console.aws.amazon.com/s3/buckets/${bucket}?prefix=${encodedPrefix}&showversions=false`;
}
//# sourceMappingURL=data:application/json;base64,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