"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const markdown_render_1 = require("jsii-docgen/lib/docgen/render/markdown-render");
const caching_1 = require("../../caching");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const util_1 = require("./util");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                     ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const created = new Array();
        const deleted = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        console.log(`Fetching package: ${event.package.key}`);
        const tarballExists = await aws.s3ObjectExists(event.bucket, event.package.key);
        if (!tarballExists) {
            throw new Error(`Tarball does not exist at key ${event.package.key} in bucket ${event.bucket}.`);
        }
        const readStream = aws.s3().getObject({ Bucket: event.bucket, Key: event.package.key }).createReadStream();
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'packages-'));
        const tarball = path.join(tmpdir, 'package.tgz');
        await util_1.writeFile(tarball, readStream);
        const uploads = new Map();
        const deletions = new Map();
        let unprocessable = false;
        function markPackage(e, marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const upload = uploadFile(event.bucket, key, event.assembly.versionId, Buffer.from(e.message));
            uploads.set(key, upload);
        }
        async function unmarkPackage(marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const marked = await aws.s3ObjectExists(event.bucket, key);
            if (!marked) {
                return;
            }
            const deletion = deleteFile(event.bucket, key);
            deletions.set(key, deletion);
        }
        console.log(`Generating documentation for ${packageFqn}...`);
        try {
            const docs = await docgen.Documentation.forPackage(tarball, { name: assembly.name });
            // if the package used to not be installable, remove the marker for it.
            await unmarkPackage(constants.UNINSTALLABLE_PACKAGE_SUFFIX);
            for (const language of language_1.DocumentationLanguage.ALL) {
                if (event.languages && !event.languages[language.toString()]) {
                    console.log(`Skipping language ${language} as it was not requested!`);
                    continue;
                }
                const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
                    metrics.setDimensions();
                    metrics.setNamespace(constants_1.METRICS_NAMESPACE);
                    async function renderAndDispatch(submodule) {
                        try {
                            console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                            const json = await docs.toJson({
                                submodule,
                                language: docgen.Language.fromString(lang.name),
                            });
                            const jsonPage = Buffer.from(json.render(null, 2));
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, jsonPage.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: jsonBody, contentEncoding: jsonContentEncoding } = compress_content_lambda_shared_1.compressContent(jsonPage);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, jsonBody.length, aws_embedded_metrics_1.Unit.Bytes);
                            const jsonKey = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule, 'json'));
                            console.log(`Uploading ${jsonKey}`);
                            const jsonUpload = uploadFile(event.bucket, jsonKey, event.assembly.versionId, jsonBody, jsonContentEncoding);
                            uploads.set(jsonKey, jsonUpload);
                            const markdown = markdown_render_1.MarkdownRenderer.fromSchema(json.content, {
                                anchorFormatter,
                                linkFormatter: linkFormatter(lang),
                            });
                            const page = Buffer.from(markdown.render());
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: body, contentEncoding } = compress_content_lambda_shared_1.compressContent(page);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                            const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule, 'md'));
                            console.log(`Uploading ${key}`);
                            const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                            uploads.set(key, upload);
                            // if the package used to have a corrupt assembly, remove the marker for it.
                            await unmarkPackage(constants.corruptAssemblyKeySuffix(language, submodule, 'md'));
                        }
                        catch (e) {
                            if (e instanceof docgen.LanguageNotSupportedError) {
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule, 'json'));
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule, 'md'));
                            }
                            else if (e instanceof docgen.CorruptedAssemblyError) {
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule, 'json'));
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule, 'md'));
                                unprocessable = true;
                            }
                            else {
                                throw e;
                            }
                        }
                    }
                    await renderAndDispatch();
                    for (const submodule of submodules) {
                        await renderAndDispatch(submodule);
                    }
                });
                await generateDocs(language);
            }
        }
        catch (error) {
            if (error instanceof docgen.UnInstallablePackageError) {
                markPackage(error, constants.UNINSTALLABLE_PACKAGE_SUFFIX);
                unprocessable = true;
            }
            else {
                throw error;
            }
        }
        for (const [key, upload] of uploads.entries()) {
            const response = await upload;
            created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
        }
        for (const [key, deletion] of deletions.entries()) {
            const response = await deletion;
            deleted.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished deleting ${key} (Version ID: ${response.VersionId})`);
        }
        if (unprocessable) {
            // the message here doesn't matter, we only use the error name
            // to divert this message away from the DLQ.
            const error = new Error();
            error.name = constants.UNPROCESSABLE_PACKAGE_ERROR_NAME;
        }
        // output must be compressed to satisfy 262,144 byte limit of SendTaskSuccess command
        const s3OKey = (s3Obj) => s3Obj.key;
        return { created: created.map(s3OKey), deleted: deleted.map(s3OKey) };
    });
}
exports.handler = handler;
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    const contentType = key.endsWith('.md') ?
        'text/markdown; charset=UTF-8' :
        key.endsWith('.json') ?
            'application/json; charset=UTF-8' :
            'application/octet-stream';
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: caching_1.CacheStrategy.default().toString(),
        ContentEncoding: contentEncoding,
        ContentType: contentType,
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
function deleteFile(bucket, key) {
    return aws.s3().deleteObject({
        Bucket: bucket,
        Key: key,
    }).promise();
}
function anchorFormatter(type) {
    const name = getAssemblyRelativeName(type); // BucketProps.Initializer.parameter.accessControl
    const [base, ...rest] = name.split('.');
    if (rest.length > 0) {
        return sanitize(rest.join('.')); // Initializer.parameter.accessControl
    }
    else {
        return sanitize(base);
    }
}
function linkFormatter(lang) {
    const formatter = (type) => {
        const name = getAssemblyRelativeName(type); // BucketProps.Initializer.parameter.accessControl
        const [baseName, ...rest] = name.split('.');
        const hash = '#' + rest.join('.'); // #Initializer.parameter.accessControl
        const langParam = `?lang=${lang.toString()}`;
        const submoduleParam = type.submodule ? `&submodule=${type.submodule}` : '';
        return `<a href="/packages/${type.packageName}/v/${type.packageVersion}/api/${baseName}${langParam}${submoduleParam}${hash}">${type.displayName}</a>`;
    };
    return formatter;
}
/**
 * Converts a type's id to an assembly-relative version, e.g.:
 * `aws-cdk-lib.aws_s3.Bucket.parameter.accessControl` => `Bucket.parameter.accessControl`
 */
function getAssemblyRelativeName(type) {
    let name = type.id;
    if (!name.startsWith(type.packageName)) {
        throw new Error(`Expected first part of "${type.id}" to start with "${type.packageName}".`);
    }
    name = name.slice(type.packageName.length + 1); // remove "aws-cdk-lib.""
    if (type.submodule) {
        if (!name.startsWith(type.submodule)) {
            throw new Error(`Expected second part of "${type.id}" to start with "${type.submodule}".`);
        }
        name = name.slice(type.submodule.length + 1); // remove "aws_s3."
    }
    return name;
}
;
function sanitize(str) {
    // HTML5 allows any characters in IDs except whitespace
    return str.replace(/ /g, '-');
}
//# sourceMappingURL=data:application/json;base64,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