"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseFargateServices = exports.LoadBalancerAccessibility = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const cdk = require("@aws-cdk/core");
const cdknag = require("./cdknag");
const common_functions_1 = require("./common/common-functions");
/**
 * The load balancer accessibility.
 *
 * @stability stable
 */
var LoadBalancerAccessibility;
(function (LoadBalancerAccessibility) {
    LoadBalancerAccessibility["EXTERNAL_ONLY"] = "EXTERNAL_ONLY";
    LoadBalancerAccessibility["INTERNAL_ONLY"] = "INTERNAL_ONLY";
})(LoadBalancerAccessibility = exports.LoadBalancerAccessibility || (exports.LoadBalancerAccessibility = {}));
/**
 * @stability stable
 */
class BaseFargateServices extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        /**
         * @stability stable
         */
        this.zoneName = '';
        /**
         * @stability stable
         */
        this.hasExternalLoadBalancer = false;
        /**
         * @stability stable
         */
        this.hasInternalLoadBalancer = false;
        /**
         * @stability stable
         */
        this.vpcSubnets = { subnetType: ec2.SubnetType.PRIVATE_WITH_NAT };
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : common_functions_1.getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.accessibility != LoadBalancerAccessibility.INTERNAL_ONLY) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.accessibility != LoadBalancerAccessibility.EXTERNAL_ONLY) {
                this.hasInternalLoadBalancer = true;
            }
        });
        const cluster = new ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
            containerInsights: true,
            executeCommandConfiguration: {
                logging: ecs.ExecuteCommandLogging.DEFAULT,
            },
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        props.tasks.forEach(t => {
            var _b, _c, _d;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
            });
            this.service.push(svc);
        });
        // Route53
        this.zoneName = (_d = (_c = props.route53Ops) === null || _c === void 0 ? void 0 : _c.zoneName) !== null && _d !== void 0 ? _d : 'svc.local';
        // ensure the dependency
        const cp = this.node.tryFindChild('Cluster');
        this.service.forEach(s => {
            s.node.addDependency(cp);
        });
        // add solution ID for the stack
        if (!cdk.Stack.of(this).templateOptions.description) {
            cdk.Stack.of(this).templateOptions.description = '(SO8030) - AWS CDK stack with serverless-container-constructs';
        }
        props.tasks.forEach(t => {
            var _b, _c;
            let cfnPolicy = (_b = t.task.executionRole) === null || _b === void 0 ? void 0 : _b.node.tryFindChild('DefaultPolicy');
            cdknag.Suppress.iamPolicy(cfnPolicy, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'ecr:GetAuthorizationToken requires wildcard resource',
                },
            ]);
            cfnPolicy = (_c = t.task.taskRole) === null || _c === void 0 ? void 0 : _c.node.tryFindChild('DefaultPolicy');
            cdknag.Suppress.iamPolicy(cfnPolicy, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'task role with ECS exec support requires wildcard resource for ssmmessages. see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html',
                },
            ]);
        });
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.BaseFargateServices = BaseFargateServices;
_a = JSII_RTTI_SYMBOL_1;
BaseFargateServices[_a] = { fqn: "serverless-container-constructs.BaseFargateServices", version: "0.1.4" };
//# sourceMappingURL=data:application/json;base64,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