"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiSpecProject = void 0;
const path = require("path");
const projen_1 = require("projen");
const parsed_spec_1 = require("./components/parsed-spec");
/**
 * Project containing the OpenAPI spec, and a parsed spec for use by the CDK construct
 */
class OpenApiSpecProject extends projen_1.Project {
    constructor(options) {
        super(options);
        // Store whether we've synthesized the project
        this.synthed = false;
        this.specFileName = options.specFileName ?? "spec.yaml";
        this.parsedSpecFileName = options.parsedSpecFileName ?? "parsed-spec.json";
        if (!this.parsedSpecFileName.endsWith(".json")) {
            throw new Error("Parsed spec file must end with .json");
        }
        this.specPath = path.join(this.outdir, this.specFileName);
        this.parsedSpecPath = path.join(this.outdir, this.parsedSpecFileName);
        // Create a sample OpenAPI spec yaml if not defined
        new projen_1.SampleFile(this, this.specFileName, {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "spec.yaml"),
        });
        // Parse the spec to produce a consolidated, bundled spec which can be read by cdk constructs or other tooling
        new parsed_spec_1.ParsedSpec(this, {
            specPath: this.specPath,
            outputPath: this.parsedSpecPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            return;
        }
        super.synth();
        this.synthed = true;
    }
}
exports.OpenApiSpecProject = OpenApiSpecProject;
//# sourceMappingURL=data:application/json;base64,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