"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareApiSpec = exports.concatMethodAndPath = void 0;
const constants_1 = require("./constants");
/**
 * Serialise a method and path into a single string
 */
exports.concatMethodAndPath = ({ method, path }) => `${method.toLowerCase()}||${path.toLowerCase()}`;
/**
 * Create the OpenAPI definition with api gateway extensions for the given authorizer
 * @param methodAuthorizer the authorizer used for the method
 */
const applyMethodAuthorizer = (methodAuthorizer) => {
    if (methodAuthorizer) {
        if (methodAuthorizer.authorizerId === constants_1.DefaultAuthorizerIds.NONE) {
            return {
                security: [],
                "x-amazon-apigateway-auth": {
                    type: "NONE",
                },
            };
        }
        else {
            return {
                security: [
                    {
                        [methodAuthorizer.authorizerId]: methodAuthorizer.authorizationScopes || [],
                    },
                ],
            };
        }
    }
    return {};
};
/**
 * Adds API Gateway integrations and auth to the given operation
 */
const applyMethodIntegration = (path, method, { integrations, corsOptions }, operation, getOperationName) => {
    const operationName = getOperationName({ method, path });
    if (!(operationName in integrations)) {
        throw new Error(`Missing required integration for operation ${operationName} (${method} ${path})`);
    }
    const { methodAuthorizer, functionInvocationUri } = integrations[operationName];
    validateAuthorizerReference(methodAuthorizer, operation.security, operationName);
    return {
        ...operation,
        responses: Object.fromEntries(Object.entries(operation.responses).map(([statusCode, response]) => [
            statusCode,
            {
                ...response,
                headers: {
                    ...(corsOptions ? getCorsHeaderDefinitions() : {}),
                    // TODO: Consider following response header references
                    ...response.headers,
                },
            },
        ])),
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-integration.html
        "x-amazon-apigateway-integration": {
            type: "AWS_PROXY",
            httpMethod: "POST",
            uri: functionInvocationUri,
            passthroughBehavior: "WHEN_NO_MATCH",
        },
        ...applyMethodAuthorizer(methodAuthorizer),
    };
};
const getCorsHeaderDefinitions = () => ({
    "Access-Control-Allow-Origin": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Methods": {
        schema: { type: "string" },
    },
    "Access-Control-Allow-Headers": {
        schema: { type: "string" },
    },
});
const generateCorsResponseHeaders = (corsOptions) => ({
    "Access-Control-Allow-Headers": `'${corsOptions.allowHeaders.join(",")}'`,
    "Access-Control-Allow-Methods": `'${corsOptions.allowMethods.join(",")}'`,
    "Access-Control-Allow-Origin": `'${corsOptions.allowOrigins.join(",")}'`,
});
const generateCorsResponseParameters = (corsOptions, prefix = "method.response.header") => Object.fromEntries(Object.entries(generateCorsResponseHeaders(corsOptions)).map(([header, value]) => [`${prefix}.${header}`, value]));
/**
 * Generates an "options" method with no auth to respond with the appropriate headers if cors is enabled
 */
const generateCorsOptionsMethod = (pathItem, { corsOptions }) => {
    // Do not generate if already manually defined, or cors not enabled
    if (constants_1.HttpMethods.OPTIONS in pathItem || !corsOptions) {
        return {};
    }
    const statusCode = corsOptions.statusCode;
    return {
        [constants_1.HttpMethods.OPTIONS]: {
            summary: "CORS Support",
            description: "Enable CORS by returning the correct headers",
            responses: {
                [`${statusCode}`]: {
                    description: "Default response for CORS method",
                    headers: getCorsHeaderDefinitions(),
                    content: {},
                },
            },
            // @ts-ignore Ignore apigateway extensions which are not part of default openapi spec type
            "x-amazon-apigateway-integration": {
                type: "mock",
                requestTemplates: {
                    "application/json": `{"statusCode": ${statusCode}}`,
                },
                responses: {
                    default: {
                        statusCode: `${statusCode}`,
                        responseParameters: generateCorsResponseParameters(corsOptions),
                        responseTemplates: {
                            "application/json": "{}",
                        },
                    },
                },
            },
        },
    };
};
/**
 * Prepares a given api path by adding integrations, configuring auth
 */
const preparePathSpec = (path, pathItem, options, getOperationName) => {
    return {
        ...pathItem,
        ...Object.fromEntries(Object.values(constants_1.HttpMethods)
            .filter((method) => pathItem[method])
            .map((method) => [
            method,
            applyMethodIntegration(path, method, options, pathItem[method], getOperationName),
        ])),
        // Generate an 'options' method required for CORS preflight requests if cors is enabled
        ...generateCorsOptionsMethod(pathItem, options),
    };
};
/**
 * Return whether the given OpenAPI object is a reference object
 */
const isRef = (obj) => "$ref" in obj;
/**
 * Validate the construct security schemes against the security schemes in the original spec.
 * Construct-defined authorizers always override those in the spec if they have the same ID, however we validate that
 * we are not overriding an authorizer of a different type to avoid mistakes/mismatches between the spec and the
 * construct.
 * @param constructSecuritySchemes security schemes generated from the construct authorizers
 * @param existingSpecSecuritySchemes security schemes already defined in the spec
 */
const validateSecuritySchemes = (constructSecuritySchemes, existingSpecSecuritySchemes) => {
    if (existingSpecSecuritySchemes) {
        const constructSecuritySchemeIds = new Set(Object.keys(constructSecuritySchemes));
        const existingSecuritySchemeIds = new Set(Object.keys(existingSpecSecuritySchemes));
        const overlappingSecuritySchemeIds = [...constructSecuritySchemeIds].filter((id) => existingSecuritySchemeIds.has(id));
        // Any overlapping security schemes (defined in both the spec (or source smithy model) and the construct) must be of the same type.
        // The one defined in the construct will take precedence since a custom/cognito authorizer can have a resolved arn in the construct,
        // and we allow usage in the model as a forward definition with blank arn.
        overlappingSecuritySchemeIds.forEach((schemeId) => {
            if (!isRef(existingSpecSecuritySchemes[schemeId])) {
                const existingScheme = existingSpecSecuritySchemes[schemeId];
                if (constructSecuritySchemes[schemeId].type !== existingScheme.type) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructSecuritySchemes[schemeId].type} in construct but ${existingScheme.type} in OpenAPI spec or Smithy model.`);
                }
                const constructApiGatewayAuthType = constructSecuritySchemes[schemeId]["x-amazon-apigateway-authtype"];
                const existingApiGatewayAuthType = existingScheme["x-amazon-apigateway-authtype"];
                if (constructApiGatewayAuthType !== existingApiGatewayAuthType) {
                    throw new Error(`Security scheme with id ${schemeId} was of type ${constructApiGatewayAuthType} in construct but ${existingApiGatewayAuthType} in OpenAPI spec or Smithy model.`);
                }
            }
            else {
                throw new Error(`Security scheme with id ${schemeId} is a reference in the OpenAPI spec or Smithy model which is not supported.`);
            }
        });
    }
};
/**
 * Validate the given authorizer reference (either default or at an operation level) defined in the construct against
 * those already in the spec.
 * @param constructAuthorizer the authorizer defined in the construct
 * @param existingSpecAuthorizers the authorizers already defined in the spec
 * @param operation the operation we are validating (for clearer error messages)
 */
const validateAuthorizerReference = (constructAuthorizer, existingSpecAuthorizers, operation = "Default") => {
    // Only need to validate if defined in both - if just one we'll use that.
    if (constructAuthorizer && existingSpecAuthorizers) {
        const mergedSpecAuthorizers = Object.fromEntries(existingSpecAuthorizers.flatMap((securityRequirement) => Object.keys(securityRequirement).map((id) => [
            id,
            securityRequirement[id],
        ])));
        const specAuthorizerIds = Object.keys(mergedSpecAuthorizers);
        if (specAuthorizerIds.length > 1) {
            // Spec defined multiple authorizers but the construct can only specify one
            throw new Error(`${operation} authorizers ${specAuthorizerIds
                .sort()
                .join(", ")} defined in the OpenAPI Spec or Smithy Model would be overridden by single construct authorizer ${constructAuthorizer.authorizerId}`);
        }
        else if (specAuthorizerIds.length === 1) {
            // Single authorizer - check that they have the same id
            if (specAuthorizerIds[0] !== constructAuthorizer.authorizerId) {
                throw new Error(`${operation} authorizer ${specAuthorizerIds[0]} defined in the OpenAPI Spec or Smithy Model would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
            }
            // Check that there are no differing scopes between the construct and spec
            const specScopes = new Set(mergedSpecAuthorizers[specAuthorizerIds[0]]);
            const constructScopes = new Set(constructAuthorizer.authorizationScopes);
            const differingScopes = [
                ...[...specScopes].filter((scope) => !constructScopes.has(scope)),
                ...[...constructScopes].filter((scope) => !specScopes.has(scope)),
            ];
            if (differingScopes.length > 0) {
                throw new Error(`${operation} authorizer scopes ${[...specScopes].join(", ")} defined in the OpenAPI Spec or Smithy Model differ from those in the construct (${[
                    ...constructScopes,
                ].join(", ")})`);
            }
        }
        else if (constructAuthorizer.authorizerId !== constants_1.DefaultAuthorizerIds.NONE) {
            // "security" section of spec is [] which means no auth, but the authorizer in the construct is not the "none" authorizer.
            throw new Error(`${operation} explicitly defines no auth in the OpenAPI Spec or Smithy Model which would be overridden by construct authorizer ${constructAuthorizer.authorizerId}`);
        }
    }
};
/**
 * Prepares the api spec for deployment by adding integrations, configuring auth, etc
 */
exports.prepareApiSpec = (spec, options) => {
    // Reverse lookup for the operation name given a method and path
    const operationNameByPath = Object.fromEntries(Object.entries(options.operationLookup).map(([operationName, methodAndPath]) => [
        exports.concatMethodAndPath(methodAndPath),
        operationName,
    ]));
    const getOperationName = (methodAndPath) => operationNameByPath[exports.concatMethodAndPath(methodAndPath)];
    validateSecuritySchemes(options.securitySchemes, spec.components?.securitySchemes);
    validateAuthorizerReference(options.defaultAuthorizerReference, spec.security);
    return {
        ...spec,
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-request-validators.html
        "x-amazon-apigateway-request-validators": {
            all: {
                validateRequestBody: true,
                validateRequestParameters: true,
            },
        },
        "x-amazon-apigateway-request-validator": "all",
        // https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-swagger-extensions-gateway-responses.html
        "x-amazon-apigateway-gateway-responses": {
            BAD_REQUEST_BODY: {
                statusCode: 400,
                responseTemplates: {
                    "application/json": '{"message": "$context.error.validationErrorString"}',
                },
                ...(options.corsOptions
                    ? {
                        responseParameters: generateCorsResponseParameters(options.corsOptions, "gatewayresponse.header"),
                    }
                    : {}),
            },
        },
        paths: {
            ...Object.fromEntries(Object.entries(spec.paths).map(([path, pathDetails]) => [
                path,
                preparePathSpec(path, pathDetails, options, getOperationName),
            ])),
        },
        components: {
            ...spec.components,
            securitySchemes: {
                // Apply any security schemes that already exist in the spec
                ...spec.components?.securitySchemes,
                // Construct security schemes override any in the spec with the same id
                ...options.securitySchemes,
            },
        },
        // Apply the default authorizer at the top level
        ...(options.defaultAuthorizerReference
            ? applyMethodAuthorizer(options.defaultAuthorizerReference)
            : {}),
    };
};
//# sourceMappingURL=data:application/json;base64,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