"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(projectOptions) {
        super({
            ...projectOptions,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...projectOptions?.venvOptions,
            },
            pip: true,
            poetry: false,
            pytest: false,
            setuptools: true,
        });
        const options = this.preConstruct(projectOptions);
        if (options.specFile && !path.isAbsolute(options.specFile)) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "constructs",
            "aws-cdk-lib",
            "cdk-nag",
        ].forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
            ...(options.specFile && path.isAbsolute(options.specFile)
                ? {
                    overrideSpecPath: options.specFile,
                }
                : {}),
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.PYTHON,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
            if ("addImplicitDependency" in this.parent) {
                // If we're within a monorepo, add an implicit dependency to ensure the generated python client is built first
                this.parent.addImplicitDependency(this, this.generatedPythonClient);
            }
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in spec_utils.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "spec_utils.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC_PATH = path.join(str(Path(__file__).absolute().parent), "${this.specDir}/${spec.parsedSpecFileName}")
SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Set up pytest manually since the default pytest generates tests for sample code which doesn't exist
        const pytestVersion = options.pytestOptions?.version || "6.2.1";
        this.addDevDependency(`pytest@${pytestVersion}`);
        this.testTask.exec("pytest");
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * This method provides inheritors a chance to synthesize extra resources prior to those created by this project.
     * Return any options you wish to change, other than python project options.
     */
    preConstruct(options) {
        return options;
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.10.1" };
//# sourceMappingURL=data:application/json;base64,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