"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeArtifact = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../deep-link");
const metric_utils_1 = require("../metric-utils");
const storage_1 = require("../s3/storage");
const code_artifact_forwarder_1 = require("./codeartifact/code-artifact-forwarder");
const constants_lambda_shared_1 = require("./codeartifact/constants.lambda-shared");
/**
 * (experimental) A package source that obtains package data from an npm CodeArtifact repository.
 *
 * @experimental
 */
class CodeArtifact {
    /**
     * @experimental
     */
    constructor(props) {
        this.props = props;
    }
    /**
     * (experimental) Binds the package source to a scope and target queue.
     *
     * @experimental
     */
    bind(scope, { denyList, ingestion, licenseList, monitoring, queue }) {
        const idPrefix = this.props.repository.node.path;
        const repositoryId = `${this.props.repository.attrDomainOwner}:${this.props.repository.attrDomainName}/${this.props.repository.attrName}`;
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(scope);
        const bucket = this.props.bucket || storageFactory.newBucket(scope, `${idPrefix}/StagingBucket`, {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [{ expiration: core_1.Duration.days(30) }],
        });
        bucket.grantRead(ingestion);
        const dlq = new aws_sqs_1.Queue(scope, `${idPrefix}/DLQ`, {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const forwarder = new code_artifact_forwarder_1.CodeArtifactForwarder(scope, `${idPrefix}/Forwarder`, {
            deadLetterQueue: dlq,
            description: `[${scope.node.path}/CodeArtifact/${repositoryId}] Handle CodeArtifact EventBridge events`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                QUEUE_URL: queue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.seconds(60),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(forwarder);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(forwarder);
        licenseList.grantRead(forwarder);
        queue.grantSendMessages(forwarder);
        forwarder.addToRolePolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['codeartifact:GetPackageVersionAsset'],
            resources: [
                core_1.Stack.of(scope).formatArn({
                    service: 'codeartifact',
                    resource: 'package',
                    arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                    resourceName: [
                        this.props.repository.attrDomainName,
                        this.props.repository.attrName,
                        'npm',
                        '*',
                    ].join('/'),
                }),
            ],
        }));
        const rule = new aws_events_1.Rule(scope, `${idPrefix}/EventBridge`, {
            description: `${scope.node.path}/CodeArtifact/${repositoryId}/EventBridge`,
            eventPattern: {
                source: ['aws.codeartifact'],
                detailType: ['CodeArtifact Package Version State Change'],
                detail: {
                    domainOwner: this.props.repository.attrDomainOwner,
                    domainName: this.props.repository.attrDomainName,
                    repositoryName: this.props.repository.attrName,
                    packageFormat: 'npm',
                },
            },
            targets: [new aws_events_targets_1.LambdaFunction(forwarder)],
        });
        const failureAlarm = forwarder.metricErrors().createAlarm(scope, `${idPrefix}/Forwarder/Failures`, {
            alarmName: `${scope.node.path}/CodeArtifact/${repositoryId}/Forwarder`,
            alarmDescription: [
                `The CodeArtifact fowarder for ${repositoryId} is failing`,
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(forwarder)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        monitoring.addHighSeverityAlarm(`CodeArtifact:${this.props.repository.attrName} Failures`, failureAlarm);
        const dlqNotEmptyAlarm = new aws_cloudwatch_1.MathExpression({
            expression: 'mVisible + mHidden',
            usingMetrics: {
                mVisible: dlq.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                mHidden: dlq.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(scope, `${idPrefix}/Forwarder/DLQNotEmpty`, {
            alarmName: `${scope.node.path}/CodeArtifact/${repositoryId}/DLQNotEmpty`,
            alarmDescription: [
                `The CodeArtifact fowarder for ${repositoryId} is failing`,
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(forwarder)}`,
                `Link to the dead letter queue: ${deep_link_1.sqsQueueUrl(dlq)}`,
            ].join('/n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        });
        monitoring.addLowSeverityAlarm(`CodeArtifact/${repositoryId} DLQ Not Empty`, dlqNotEmptyAlarm);
        rule.node.addDependency(failureAlarm, dlqNotEmptyAlarm);
        return {
            name: `CodeArtifact: ${repositoryId}`,
            links: [{
                    name: 'CodeArtifact',
                    url: deep_link_1.codeArtifactRepositoryUrl(this.props.repository),
                    primary: true,
                }, {
                    name: 'Forwarder Function',
                    url: deep_link_1.lambdaFunctionUrl(forwarder),
                }, {
                    name: 'Search Log group',
                    url: deep_link_1.lambdaSearchLogGroupUrl(forwarder),
                }, {
                    name: 'DLQ',
                    url: deep_link_1.sqsQueueUrl(dlq),
                }],
            dashboardWidgets: [
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(forwarder.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(forwarder.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            forwarder.metricDuration({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(15),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            dlq.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            dlq.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            dlq.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Quality Metrics',
                        left: [
                            metric_utils_1.fillMetric(this.metricNotJsiiEnabledCount({ label: 'Not a jsii package' }), 0),
                            metric_utils_1.fillMetric(this.metricIneligibleLicense({ label: 'Ineligible License' }), 0),
                            metric_utils_1.fillMetric(this.metricDenyListedCount({ label: 'Deny Listed' }), 0),
                            metric_utils_1.fillMetric(this.metricDeletedCount({ label: 'Deletion Events' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                ],
            ],
        };
    }
    /**
     * (experimental) The count of package versions that were ignored due to being in the deny list.
     *
     * @experimental
     */
    metricDenyListedCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: this.props.repository.attrDomainOwner,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "DenyListedCount" /* DENY_LISTED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The number of package versions that were ignored due to using an ineloigible license.
     *
     * @experimental
     */
    metricIneligibleLicense(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "IneligibleLicenseCount" /* INELIGIBLE_LICENSE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The number of package versions that were deleted from CodeArtifact (those events are not handled currently).
     *
     * @experimental
     */
    metricDeletedCount(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "DeletedCount" /* DELETED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The number of package versions that do not have a jsii assembly in the package.
     *
     * @experimental
     */
    metricNotJsiiEnabledCount(opts) {
        var _b;
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: {
                [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: (_b = this.props.repository.attrDomainOwner) !== null && _b !== void 0 ? _b : core_1.Aws.ACCOUNT_ID,
                [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: this.props.repository.attrDomainName,
                [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: this.props.repository.attrName,
            },
            metricName: "NotJsiiEnabledCount" /* NOT_JSII_ENABLED_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
}
exports.CodeArtifact = CodeArtifact;
_a = JSII_RTTI_SYMBOL_1;
CodeArtifact[_a] = { fqn: "construct-hub.sources.CodeArtifact", version: "0.3.185" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29kZS1hcnRpZmFjdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uL3NyYy9wYWNrYWdlLXNvdXJjZXMvY29kZS1hcnRpZmFjdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDREQUE4STtBQUU5SSxvREFBMkM7QUFDM0Msb0VBQTZEO0FBQzdELDhDQUEyRDtBQUMzRCxvREFBOEM7QUFDOUMsNENBQTZEO0FBQzdELDhDQUEwRDtBQUMxRCx3Q0FBMkU7QUFDM0UsNENBQWtIO0FBQ2xILGtEQUE2QztBQUU3QywyQ0FBaUQ7QUFDakQsb0ZBQStFO0FBQy9FLG9GQUFpSzs7Ozs7O0FBV2pLLE1BQWEsWUFBWTs7OztJQUN2QixZQUFvQyxLQUF3QjtRQUF4QixVQUFLLEdBQUwsS0FBSyxDQUFtQjtJQUM1RCxDQUFDOzs7Ozs7SUFFTSxJQUFJLENBQUMsS0FBZ0IsRUFBRSxFQUFFLFFBQVEsRUFBRSxTQUFTLEVBQUUsV0FBVyxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQTRCO1FBQzdHLE1BQU0sUUFBUSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7UUFDakQsTUFBTSxZQUFZLEdBQUcsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxlQUFlLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsY0FBYyxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBRTFJLE1BQU0sY0FBYyxHQUFHLDBCQUFnQixDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUMzRCxNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sSUFBSSxjQUFjLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxHQUFHLFFBQVEsZ0JBQWdCLEVBQUU7WUFDL0YsaUJBQWlCLEVBQUUsMEJBQWlCLENBQUMsU0FBUztZQUM5QyxVQUFVLEVBQUUsSUFBSTtZQUNoQixjQUFjLEVBQUUsQ0FBQyxFQUFFLFVBQVUsRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUM7U0FDcEQsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUU1QixNQUFNLEdBQUcsR0FBRyxJQUFJLGVBQUssQ0FBQyxLQUFLLEVBQUUsR0FBRyxRQUFRLE1BQU0sRUFBRTtZQUM5QyxVQUFVLEVBQUUseUJBQWUsQ0FBQyxXQUFXO1lBQ3ZDLGVBQWUsRUFBRSxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQztZQUNsQyxpQkFBaUIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztTQUN4QyxDQUFDLENBQUM7UUFFSCxNQUFNLFNBQVMsR0FBRyxJQUFJLCtDQUFxQixDQUFDLEtBQUssRUFBRSxHQUFHLFFBQVEsWUFBWSxFQUFFO1lBQzFFLGVBQWUsRUFBRSxHQUFHO1lBQ3BCLFdBQVcsRUFBRSxJQUFJLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxpQkFBaUIsWUFBWSwwQ0FBMEM7WUFDdkcsV0FBVyxFQUFFO2dCQUNYLG1CQUFtQixFQUFFLE9BQU87Z0JBQzVCLFdBQVcsRUFBRSxNQUFNLENBQUMsVUFBVTtnQkFDOUIsU0FBUyxFQUFFLEtBQUssQ0FBQyxRQUFRO2FBQzFCO1lBQ0QsVUFBVSxFQUFFLElBQUk7WUFDaEIsT0FBTyxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1lBQzdCLE9BQU8sRUFBRSxvQkFBTyxDQUFDLE1BQU07U0FDeEIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxDQUFDLGNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNqQyxRQUFRLGFBQVIsUUFBUSx1QkFBUixRQUFRLENBQUUsU0FBUyxDQUFDLFNBQVMsRUFBRTtRQUMvQixXQUFXLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ2pDLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNuQyxTQUFTLENBQUMsZUFBZSxDQUFDLElBQUkseUJBQWUsQ0FBQztZQUM1QyxNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO1lBQ3BCLE9BQU8sRUFBRSxDQUFDLHFDQUFxQyxDQUFDO1lBQ2hELFNBQVMsRUFBRTtnQkFDVCxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFNBQVMsQ0FBQztvQkFDeEIsT0FBTyxFQUFFLGNBQWM7b0JBQ3ZCLFFBQVEsRUFBRSxTQUFTO29CQUNuQixTQUFTLEVBQUUsZ0JBQVMsQ0FBQyxtQkFBbUI7b0JBQ3hDLFlBQVksRUFBRTt3QkFDWixJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxjQUFjO3dCQUNwQyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRO3dCQUM5QixLQUFLO3dCQUNMLEdBQUc7cUJBQ0osQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO2lCQUNaLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosTUFBTSxJQUFJLEdBQUcsSUFBSSxpQkFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLFFBQVEsY0FBYyxFQUFFO1lBQ3RELFdBQVcsRUFBRSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxpQkFBaUIsWUFBWSxjQUFjO1lBQzFFLFlBQVksRUFBRTtnQkFDWixNQUFNLEVBQUUsQ0FBQyxrQkFBa0IsQ0FBQztnQkFDNUIsVUFBVSxFQUFFLENBQUMsMkNBQTJDLENBQUM7Z0JBQ3pELE1BQU0sRUFBRTtvQkFDTixXQUFXLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsZUFBZTtvQkFDbEQsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLGNBQWM7b0JBQ2hELGNBQWMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRO29CQUM5QyxhQUFhLEVBQUUsS0FBSztpQkFDckI7YUFDRjtZQUNELE9BQU8sRUFBRSxDQUFDLElBQUksbUNBQWMsQ0FBQyxTQUFTLENBQUMsQ0FBQztTQUN6QyxDQUFDLENBQUM7UUFFSCxNQUFNLFlBQVksR0FBRyxTQUFTLENBQUMsWUFBWSxFQUFFLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxHQUFHLFFBQVEscUJBQXFCLEVBQUU7WUFDakcsU0FBUyxFQUFFLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLGlCQUFpQixZQUFZLFlBQVk7WUFDdEUsZ0JBQWdCLEVBQUU7Z0JBQ2hCLGlDQUFpQyxZQUFZLGFBQWE7Z0JBQzFELEVBQUU7Z0JBQ0YsZ0NBQWdDLDZCQUFpQixDQUFDLFNBQVMsQ0FBQyxFQUFFO2FBQy9ELENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztZQUNaLGtCQUFrQixFQUFFLG1DQUFrQixDQUFDLGtDQUFrQztZQUN6RSxpQkFBaUIsRUFBRSxDQUFDO1lBQ3BCLFNBQVMsRUFBRSxDQUFDO1lBQ1osZ0JBQWdCLEVBQUUsaUNBQWdCLENBQUMsT0FBTztTQUMzQyxDQUFDLENBQUM7UUFDSCxVQUFVLENBQUMsb0JBQW9CLENBQUMsZ0JBQWdCLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLFFBQVEsV0FBVyxFQUFFLFlBQVksQ0FBQyxDQUFDO1FBRXpHLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSwrQkFBYyxDQUFDO1lBQzFDLFVBQVUsRUFBRSxvQkFBb0I7WUFDaEMsWUFBWSxFQUFFO2dCQUNaLFFBQVEsRUFBRSxHQUFHLENBQUMsd0NBQXdDLENBQUMsRUFBRSxNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUN2RixPQUFPLEVBQUUsR0FBRyxDQUFDLDJDQUEyQyxDQUFDLEVBQUUsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzthQUMxRjtTQUNGLENBQUMsQ0FBQyxXQUFXLENBQUMsS0FBSyxFQUFFLEdBQUcsUUFBUSx3QkFBd0IsRUFBRTtZQUN6RCxTQUFTLEVBQUUsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksaUJBQWlCLFlBQVksY0FBYztZQUN4RSxnQkFBZ0IsRUFBRTtnQkFDaEIsaUNBQWlDLFlBQVksYUFBYTtnQkFDMUQsRUFBRTtnQkFDRixnQ0FBZ0MsNkJBQWlCLENBQUMsU0FBUyxDQUFDLEVBQUU7Z0JBQzlELGtDQUFrQyx1QkFBVyxDQUFDLEdBQUcsQ0FBQyxFQUFFO2FBQ3JELENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztZQUNaLGtCQUFrQixFQUFFLG1DQUFrQixDQUFDLGtDQUFrQztZQUN6RSxpQkFBaUIsRUFBRSxDQUFDO1lBQ3BCLFNBQVMsRUFBRSxDQUFDO1lBQ1osZ0JBQWdCLEVBQUUsaUNBQWdCLENBQUMsYUFBYTtTQUNqRCxDQUFDLENBQUM7UUFDSCxVQUFVLENBQUMsbUJBQW1CLENBQUMsZ0JBQWdCLFlBQVksZ0JBQWdCLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUUvRixJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUV4RCxPQUFPO1lBQ0wsSUFBSSxFQUFFLGlCQUFpQixZQUFZLEVBQUU7WUFDckMsS0FBSyxFQUFFLENBQUM7b0JBQ04sSUFBSSxFQUFFLGNBQWM7b0JBQ3BCLEdBQUcsRUFBRSxxQ0FBeUIsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQztvQkFDckQsT0FBTyxFQUFFLElBQUk7aUJBQ2QsRUFBRTtvQkFDRCxJQUFJLEVBQUUsb0JBQW9CO29CQUMxQixHQUFHLEVBQUUsNkJBQWlCLENBQUMsU0FBUyxDQUFDO2lCQUNsQyxFQUFFO29CQUNELElBQUksRUFBRSxrQkFBa0I7b0JBQ3hCLEdBQUcsRUFBRSxtQ0FBdUIsQ0FBQyxTQUFTLENBQUM7aUJBQ3hDLEVBQUU7b0JBQ0QsSUFBSSxFQUFFLEtBQUs7b0JBQ1gsR0FBRyxFQUFFLHVCQUFXLENBQUMsR0FBRyxDQUFDO2lCQUN0QixDQUFDO1lBQ0YsZ0JBQWdCLEVBQUU7Z0JBQ2hCO29CQUNFLElBQUksNEJBQVcsQ0FBQzt3QkFDZCxNQUFNLEVBQUUsQ0FBQzt3QkFDVCxLQUFLLEVBQUUsRUFBRTt3QkFDVCxLQUFLLEVBQUUsaUJBQWlCO3dCQUN4QixJQUFJLEVBQUU7NEJBQ0oseUJBQVUsQ0FBQyxTQUFTLENBQUMsaUJBQWlCLENBQUMsRUFBRSxLQUFLLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQzs0QkFDakUseUJBQVUsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLEVBQUUsS0FBSyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7eUJBQ3hEO3dCQUNELFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7d0JBQ3JCLEtBQUssRUFBRTs0QkFDTCxTQUFTLENBQUMsY0FBYyxDQUFDLEVBQUUsS0FBSyxFQUFFLFVBQVUsRUFBRSxDQUFDO3lCQUNoRDt3QkFDRCxVQUFVLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO3dCQUN0QixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7cUJBQzdCLENBQUM7b0JBQ0YsSUFBSSw0QkFBVyxDQUFDO3dCQUNkLE1BQU0sRUFBRSxDQUFDO3dCQUNULEtBQUssRUFBRSxFQUFFO3dCQUNULEtBQUssRUFBRSxtQkFBbUI7d0JBQzFCLElBQUksRUFBRTs0QkFDSixHQUFHLENBQUMsd0NBQXdDLENBQUMsRUFBRSxLQUFLLEVBQUUsa0JBQWtCLEVBQUUsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzs0QkFDeEcsR0FBRyxDQUFDLDJDQUEyQyxDQUFDLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7eUJBQzNHO3dCQUNELFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxDQUFDLEVBQUU7d0JBQ3JCLEtBQUssRUFBRTs0QkFDTCxHQUFHLENBQUMsbUNBQW1DLENBQUMsRUFBRSxLQUFLLEVBQUUsb0JBQW9CLEVBQUUsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzt5QkFDdEc7d0JBQ0QsVUFBVSxFQUFFLEVBQUUsR0FBRyxFQUFFLENBQUMsRUFBRTtxQkFDdkIsQ0FBQztpQkFDSDtnQkFDRDtvQkFDRSxJQUFJLDRCQUFXLENBQUM7d0JBQ2QsTUFBTSxFQUFFLENBQUM7d0JBQ1QsS0FBSyxFQUFFLEVBQUU7d0JBQ1QsS0FBSyxFQUFFLGlCQUFpQjt3QkFDeEIsSUFBSSxFQUFFOzRCQUNKLHlCQUFVLENBQUMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLEVBQUUsS0FBSyxFQUFFLG9CQUFvQixFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7NEJBQzlFLHlCQUFVLENBQUMsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEVBQUUsS0FBSyxFQUFFLG9CQUFvQixFQUFFLENBQUMsRUFBRSxDQUFDLENBQUM7NEJBQzVFLHlCQUFVLENBQUMsSUFBSSxDQUFDLHFCQUFxQixDQUFDLEVBQUUsS0FBSyxFQUFFLGFBQWEsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDOzRCQUNuRSx5QkFBVSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFLEtBQUssRUFBRSxpQkFBaUIsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO3lCQUNyRTt3QkFDRCxTQUFTLEVBQUUsRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFO3FCQUN0QixDQUFDO2lCQUNIO2FBQ0Y7U0FDRixDQUFDO0lBQ0osQ0FBQzs7Ozs7O0lBR00scUJBQXFCLENBQUMsSUFBb0I7UUFDL0MsT0FBTyxJQUFJLHVCQUFNLENBQUM7WUFDaEIsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzNCLFNBQVMsRUFBRSwwQkFBUyxDQUFDLEdBQUc7WUFDeEIsR0FBRyxJQUFJO1lBQ1AsYUFBYSxFQUFFO2dCQUNiLENBQUMsZ0RBQXNCLENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxlQUFlO2dCQUMvRCxDQUFDLCtDQUFxQixDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsY0FBYztnQkFDN0QsQ0FBQyxtREFBeUIsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLFFBQVE7YUFDNUQ7WUFDRCxVQUFVLDJDQUE4QjtZQUN4QyxTQUFTLEVBQUUsMkNBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7OztJQUdNLHVCQUF1QixDQUFDLElBQW9COztRQUNqRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxhQUFhLEVBQUU7Z0JBQ2IsQ0FBQyxnREFBc0IsQ0FBQyxRQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLGVBQWUsbUNBQUksVUFBRyxDQUFDLFVBQVU7Z0JBQ2pGLENBQUMsK0NBQXFCLENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxjQUFjO2dCQUM3RCxDQUFDLG1EQUF5QixDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsUUFBUTthQUM1RDtZQUNELFVBQVUsbURBQStCO1lBQ3pDLFNBQVMsRUFBRSwyQ0FBaUI7U0FDN0IsQ0FBQyxDQUFDO0lBQ0wsQ0FBQzs7Ozs7O0lBR00sa0JBQWtCLENBQUMsSUFBb0I7O1FBQzVDLE9BQU8sSUFBSSx1QkFBTSxDQUFDO1lBQ2hCLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUMzQixTQUFTLEVBQUUsMEJBQVMsQ0FBQyxHQUFHO1lBQ3hCLEdBQUcsSUFBSTtZQUNQLGFBQWEsRUFBRTtnQkFDYixDQUFDLGdEQUFzQixDQUFDLFFBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsZUFBZSxtQ0FBSSxVQUFHLENBQUMsVUFBVTtnQkFDakYsQ0FBQywrQ0FBcUIsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLGNBQWM7Z0JBQzdELENBQUMsbURBQXlCLENBQUMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRO2FBQzVEO1lBQ0QsVUFBVSxvQ0FBMEI7WUFDcEMsU0FBUyxFQUFFLDJDQUFpQjtTQUM3QixDQUFDLENBQUM7SUFDTCxDQUFDOzs7Ozs7SUFHTSx5QkFBeUIsQ0FBQyxJQUFvQjs7UUFDbkQsT0FBTyxJQUFJLHVCQUFNLENBQUM7WUFDaEIsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzNCLFNBQVMsRUFBRSwwQkFBUyxDQUFDLEdBQUc7WUFDeEIsR0FBRyxJQUFJO1lBQ1AsYUFBYSxFQUFFO2dCQUNiLENBQUMsZ0RBQXNCLENBQUMsUUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxlQUFlLG1DQUFJLFVBQUcsQ0FBQyxVQUFVO2dCQUNqRixDQUFDLCtDQUFxQixDQUFDLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsY0FBYztnQkFDN0QsQ0FBQyxtREFBeUIsQ0FBQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLFFBQVE7YUFDNUQ7WUFDRCxVQUFVLG9EQUFtQztZQUM3QyxTQUFTLEVBQUUsMkNBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBNU9ILG9DQTZPQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbXBhcmlzb25PcGVyYXRvciwgR3JhcGhXaWRnZXQsIE1hdGhFeHByZXNzaW9uLCBNZXRyaWMsIE1ldHJpY09wdGlvbnMsIFN0YXRpc3RpYywgVHJlYXRNaXNzaW5nRGF0YSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCB7IENmblJlcG9zaXRvcnkgfSBmcm9tICdAYXdzLWNkay9hd3MtY29kZWFydGlmYWN0JztcbmltcG9ydCB7IFJ1bGUgfSBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzJztcbmltcG9ydCB7IExhbWJkYUZ1bmN0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWV2ZW50cy10YXJnZXRzJztcbmltcG9ydCB7IEVmZmVjdCwgUG9saWN5U3RhdGVtZW50IH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBUcmFjaW5nIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBCbG9ja1B1YmxpY0FjY2VzcywgSUJ1Y2tldCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBRdWV1ZSwgUXVldWVFbmNyeXB0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLXNxcyc7XG5pbXBvcnQgeyBBcm5Gb3JtYXQsIEF3cywgQ29uc3RydWN0LCBEdXJhdGlvbiwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IGNvZGVBcnRpZmFjdFJlcG9zaXRvcnlVcmwsIGxhbWJkYUZ1bmN0aW9uVXJsLCBsYW1iZGFTZWFyY2hMb2dHcm91cFVybCwgc3FzUXVldWVVcmwgfSBmcm9tICcuLi9kZWVwLWxpbmsnO1xuaW1wb3J0IHsgZmlsbE1ldHJpYyB9IGZyb20gJy4uL21ldHJpYy11dGlscyc7XG5pbXBvcnQgdHlwZSB7IElQYWNrYWdlU291cmNlLCBQYWNrYWdlU291cmNlQmluZE9wdGlvbnMsIFBhY2thZ2VTb3VyY2VCaW5kUmVzdWx0IH0gZnJvbSAnLi4vcGFja2FnZS1zb3VyY2UnO1xuaW1wb3J0IHsgUzNTdG9yYWdlRmFjdG9yeSB9IGZyb20gJy4uL3MzL3N0b3JhZ2UnO1xuaW1wb3J0IHsgQ29kZUFydGlmYWN0Rm9yd2FyZGVyIH0gZnJvbSAnLi9jb2RlYXJ0aWZhY3QvY29kZS1hcnRpZmFjdC1mb3J3YXJkZXInO1xuaW1wb3J0IHsgTUVUUklDU19OQU1FU1BBQ0UsIE1ldHJpY05hbWUsIERPTUFJTl9OQU1FX0RJTUVOU0lPTiwgRE9NQUlOX09XTkVSX0RJTUVOU0lPTiwgUkVQT1NJVE9SWV9OQU1FX0RJTUVOU0lPTiB9IGZyb20gJy4vY29kZWFydGlmYWN0L2NvbnN0YW50cy5sYW1iZGEtc2hhcmVkJztcblxuZXhwb3J0IGludGVyZmFjZSBDb2RlQXJ0aWZhY3RQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcmVwb3NpdG9yeTogQ2ZuUmVwb3NpdG9yeTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidWNrZXQ/OiBJQnVja2V0O1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBDb2RlQXJ0aWZhY3QgaW1wbGVtZW50cyBJUGFja2FnZVNvdXJjZSB7XG4gIHB1YmxpYyBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBDb2RlQXJ0aWZhY3RQcm9wcykge1xuICB9XG5cbiAgcHVibGljIGJpbmQoc2NvcGU6IENvbnN0cnVjdCwgeyBkZW55TGlzdCwgaW5nZXN0aW9uLCBsaWNlbnNlTGlzdCwgbW9uaXRvcmluZywgcXVldWUgfTogUGFja2FnZVNvdXJjZUJpbmRPcHRpb25zKTogUGFja2FnZVNvdXJjZUJpbmRSZXN1bHQge1xuICAgIGNvbnN0IGlkUHJlZml4ID0gdGhpcy5wcm9wcy5yZXBvc2l0b3J5Lm5vZGUucGF0aDtcbiAgICBjb25zdCByZXBvc2l0b3J5SWQgPSBgJHt0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ckRvbWFpbk93bmVyfToke3RoaXMucHJvcHMucmVwb3NpdG9yeS5hdHRyRG9tYWluTmFtZX0vJHt0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ck5hbWV9YDtcblxuICAgIGNvbnN0IHN0b3JhZ2VGYWN0b3J5ID0gUzNTdG9yYWdlRmFjdG9yeS5nZXRPckNyZWF0ZShzY29wZSk7XG4gICAgY29uc3QgYnVja2V0ID0gdGhpcy5wcm9wcy5idWNrZXQgfHwgc3RvcmFnZUZhY3RvcnkubmV3QnVja2V0KHNjb3BlLCBgJHtpZFByZWZpeH0vU3RhZ2luZ0J1Y2tldGAsIHtcbiAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBCbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgICAgbGlmZWN5Y2xlUnVsZXM6IFt7IGV4cGlyYXRpb246IER1cmF0aW9uLmRheXMoMzApIH1dLFxuICAgIH0pO1xuICAgIGJ1Y2tldC5ncmFudFJlYWQoaW5nZXN0aW9uKTtcblxuICAgIGNvbnN0IGRscSA9IG5ldyBRdWV1ZShzY29wZSwgYCR7aWRQcmVmaXh9L0RMUWAsIHtcbiAgICAgIGVuY3J5cHRpb246IFF1ZXVlRW5jcnlwdGlvbi5LTVNfTUFOQUdFRCxcbiAgICAgIHJldGVudGlvblBlcmlvZDogRHVyYXRpb24uZGF5cygxNCksXG4gICAgICB2aXNpYmlsaXR5VGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgfSk7XG5cbiAgICBjb25zdCBmb3J3YXJkZXIgPSBuZXcgQ29kZUFydGlmYWN0Rm9yd2FyZGVyKHNjb3BlLCBgJHtpZFByZWZpeH0vRm9yd2FyZGVyYCwge1xuICAgICAgZGVhZExldHRlclF1ZXVlOiBkbHEsXG4gICAgICBkZXNjcmlwdGlvbjogYFske3Njb3BlLm5vZGUucGF0aH0vQ29kZUFydGlmYWN0LyR7cmVwb3NpdG9yeUlkfV0gSGFuZGxlIENvZGVBcnRpZmFjdCBFdmVudEJyaWRnZSBldmVudHNgLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgQVdTX0VNRl9FTlZJUk9OTUVOVDogJ0xvY2FsJyxcbiAgICAgICAgQlVDS0VUX05BTUU6IGJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICBRVUVVRV9VUkw6IHF1ZXVlLnF1ZXVlVXJsLFxuICAgICAgfSxcbiAgICAgIG1lbW9yeVNpemU6IDEwMjQsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKDYwKSxcbiAgICAgIHRyYWNpbmc6IFRyYWNpbmcuQUNUSVZFLFxuICAgIH0pO1xuICAgIGJ1Y2tldC5ncmFudFJlYWRXcml0ZShmb3J3YXJkZXIpO1xuICAgIGRlbnlMaXN0Py5ncmFudFJlYWQoZm9yd2FyZGVyKTtcbiAgICBsaWNlbnNlTGlzdC5ncmFudFJlYWQoZm9yd2FyZGVyKTtcbiAgICBxdWV1ZS5ncmFudFNlbmRNZXNzYWdlcyhmb3J3YXJkZXIpO1xuICAgIGZvcndhcmRlci5hZGRUb1JvbGVQb2xpY3kobmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgIGFjdGlvbnM6IFsnY29kZWFydGlmYWN0OkdldFBhY2thZ2VWZXJzaW9uQXNzZXQnXSxcbiAgICAgIHJlc291cmNlczogW1xuICAgICAgICBTdGFjay5vZihzY29wZSkuZm9ybWF0QXJuKHtcbiAgICAgICAgICBzZXJ2aWNlOiAnY29kZWFydGlmYWN0JyxcbiAgICAgICAgICByZXNvdXJjZTogJ3BhY2thZ2UnLFxuICAgICAgICAgIGFybkZvcm1hdDogQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUsXG4gICAgICAgICAgcmVzb3VyY2VOYW1lOiBbXG4gICAgICAgICAgICB0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ckRvbWFpbk5hbWUsXG4gICAgICAgICAgICB0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ck5hbWUsXG4gICAgICAgICAgICAnbnBtJywgLy8gcGFja2FnZSBmb3JtYXRcbiAgICAgICAgICAgICcqJywgLy8gbmFtZXNwYWNlL3BhY2thZ2UtbmFtZVxuICAgICAgICAgIF0uam9pbignLycpLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSkpO1xuXG4gICAgY29uc3QgcnVsZSA9IG5ldyBSdWxlKHNjb3BlLCBgJHtpZFByZWZpeH0vRXZlbnRCcmlkZ2VgLCB7XG4gICAgICBkZXNjcmlwdGlvbjogYCR7c2NvcGUubm9kZS5wYXRofS9Db2RlQXJ0aWZhY3QvJHtyZXBvc2l0b3J5SWR9L0V2ZW50QnJpZGdlYCxcbiAgICAgIGV2ZW50UGF0dGVybjoge1xuICAgICAgICBzb3VyY2U6IFsnYXdzLmNvZGVhcnRpZmFjdCddLFxuICAgICAgICBkZXRhaWxUeXBlOiBbJ0NvZGVBcnRpZmFjdCBQYWNrYWdlIFZlcnNpb24gU3RhdGUgQ2hhbmdlJ10sXG4gICAgICAgIGRldGFpbDoge1xuICAgICAgICAgIGRvbWFpbk93bmVyOiB0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ckRvbWFpbk93bmVyLFxuICAgICAgICAgIGRvbWFpbk5hbWU6IHRoaXMucHJvcHMucmVwb3NpdG9yeS5hdHRyRG9tYWluTmFtZSxcbiAgICAgICAgICByZXBvc2l0b3J5TmFtZTogdGhpcy5wcm9wcy5yZXBvc2l0b3J5LmF0dHJOYW1lLFxuICAgICAgICAgIHBhY2thZ2VGb3JtYXQ6ICducG0nLFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIHRhcmdldHM6IFtuZXcgTGFtYmRhRnVuY3Rpb24oZm9yd2FyZGVyKV0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBmYWlsdXJlQWxhcm0gPSBmb3J3YXJkZXIubWV0cmljRXJyb3JzKCkuY3JlYXRlQWxhcm0oc2NvcGUsIGAke2lkUHJlZml4fS9Gb3J3YXJkZXIvRmFpbHVyZXNgLCB7XG4gICAgICBhbGFybU5hbWU6IGAke3Njb3BlLm5vZGUucGF0aH0vQ29kZUFydGlmYWN0LyR7cmVwb3NpdG9yeUlkfS9Gb3J3YXJkZXJgLFxuICAgICAgYWxhcm1EZXNjcmlwdGlvbjogW1xuICAgICAgICBgVGhlIENvZGVBcnRpZmFjdCBmb3dhcmRlciBmb3IgJHtyZXBvc2l0b3J5SWR9IGlzIGZhaWxpbmdgLFxuICAgICAgICAnJyxcbiAgICAgICAgYExpbmsgdG8gdGhlIGxhbWJkYSBmdW5jdGlvbjogJHtsYW1iZGFGdW5jdGlvblVybChmb3J3YXJkZXIpfWAsXG4gICAgICBdLmpvaW4oJ1xcbicpLFxuICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBDb21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAzLFxuICAgICAgdGhyZXNob2xkOiAxLFxuICAgICAgdHJlYXRNaXNzaW5nRGF0YTogVHJlYXRNaXNzaW5nRGF0YS5NSVNTSU5HLFxuICAgIH0pO1xuICAgIG1vbml0b3JpbmcuYWRkSGlnaFNldmVyaXR5QWxhcm0oYENvZGVBcnRpZmFjdDoke3RoaXMucHJvcHMucmVwb3NpdG9yeS5hdHRyTmFtZX0gRmFpbHVyZXNgLCBmYWlsdXJlQWxhcm0pO1xuXG4gICAgY29uc3QgZGxxTm90RW1wdHlBbGFybSA9IG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICBleHByZXNzaW9uOiAnbVZpc2libGUgKyBtSGlkZGVuJyxcbiAgICAgIHVzaW5nTWV0cmljczoge1xuICAgICAgICBtVmlzaWJsZTogZGxxLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc1Zpc2libGUoeyBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSksXG4gICAgICAgIG1IaWRkZW46IGRscS5tZXRyaWNBcHByb3hpbWF0ZU51bWJlck9mTWVzc2FnZXNOb3RWaXNpYmxlKHsgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDEpIH0pLFxuICAgICAgfSxcbiAgICB9KS5jcmVhdGVBbGFybShzY29wZSwgYCR7aWRQcmVmaXh9L0ZvcndhcmRlci9ETFFOb3RFbXB0eWAsIHtcbiAgICAgIGFsYXJtTmFtZTogYCR7c2NvcGUubm9kZS5wYXRofS9Db2RlQXJ0aWZhY3QvJHtyZXBvc2l0b3J5SWR9L0RMUU5vdEVtcHR5YCxcbiAgICAgIGFsYXJtRGVzY3JpcHRpb246IFtcbiAgICAgICAgYFRoZSBDb2RlQXJ0aWZhY3QgZm93YXJkZXIgZm9yICR7cmVwb3NpdG9yeUlkfSBpcyBmYWlsaW5nYCxcbiAgICAgICAgJycsXG4gICAgICAgIGBMaW5rIHRvIHRoZSBsYW1iZGEgZnVuY3Rpb246ICR7bGFtYmRhRnVuY3Rpb25VcmwoZm9yd2FyZGVyKX1gLFxuICAgICAgICBgTGluayB0byB0aGUgZGVhZCBsZXR0ZXIgcXVldWU6ICR7c3FzUXVldWVVcmwoZGxxKX1gLFxuICAgICAgXS5qb2luKCcvbicpLFxuICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBDb21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICAgIGV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgICAgdGhyZXNob2xkOiAxLFxuICAgICAgdHJlYXRNaXNzaW5nRGF0YTogVHJlYXRNaXNzaW5nRGF0YS5OT1RfQlJFQUNISU5HLFxuICAgIH0pO1xuICAgIG1vbml0b3JpbmcuYWRkTG93U2V2ZXJpdHlBbGFybShgQ29kZUFydGlmYWN0LyR7cmVwb3NpdG9yeUlkfSBETFEgTm90IEVtcHR5YCwgZGxxTm90RW1wdHlBbGFybSk7XG5cbiAgICBydWxlLm5vZGUuYWRkRGVwZW5kZW5jeShmYWlsdXJlQWxhcm0sIGRscU5vdEVtcHR5QWxhcm0pO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIG5hbWU6IGBDb2RlQXJ0aWZhY3Q6ICR7cmVwb3NpdG9yeUlkfWAsXG4gICAgICBsaW5rczogW3tcbiAgICAgICAgbmFtZTogJ0NvZGVBcnRpZmFjdCcsXG4gICAgICAgIHVybDogY29kZUFydGlmYWN0UmVwb3NpdG9yeVVybCh0aGlzLnByb3BzLnJlcG9zaXRvcnkpLFxuICAgICAgICBwcmltYXJ5OiB0cnVlLFxuICAgICAgfSwge1xuICAgICAgICBuYW1lOiAnRm9yd2FyZGVyIEZ1bmN0aW9uJyxcbiAgICAgICAgdXJsOiBsYW1iZGFGdW5jdGlvblVybChmb3J3YXJkZXIpLFxuICAgICAgfSwge1xuICAgICAgICBuYW1lOiAnU2VhcmNoIExvZyBncm91cCcsXG4gICAgICAgIHVybDogbGFtYmRhU2VhcmNoTG9nR3JvdXBVcmwoZm9yd2FyZGVyKSxcbiAgICAgIH0sIHtcbiAgICAgICAgbmFtZTogJ0RMUScsXG4gICAgICAgIHVybDogc3FzUXVldWVVcmwoZGxxKSxcbiAgICAgIH1dLFxuICAgICAgZGFzaGJvYXJkV2lkZ2V0czogW1xuICAgICAgICBbXG4gICAgICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgICAgIHRpdGxlOiAnRnVuY3Rpb24gSGVhbHRoJyxcbiAgICAgICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICAgICAgZmlsbE1ldHJpYyhmb3J3YXJkZXIubWV0cmljSW52b2NhdGlvbnMoeyBsYWJlbDogJ0ludm9jYXRpb25zJyB9KSksXG4gICAgICAgICAgICAgIGZpbGxNZXRyaWMoZm9yd2FyZGVyLm1ldHJpY0Vycm9ycyh7IGxhYmVsOiAnRXJyb3JzJyB9KSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgbGVmdFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgICAgICAgcmlnaHQ6IFtcbiAgICAgICAgICAgICAgZm9yd2FyZGVyLm1ldHJpY0R1cmF0aW9uKHsgbGFiZWw6ICdEdXJhdGlvbicgfSksXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgcmlnaHRZQXhpczogeyBtaW46IDAgfSxcbiAgICAgICAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgICAgICAgfSksXG4gICAgICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgICAgIHRpdGxlOiAnRGVhZCBMZXR0ZXIgUXVldWUnLFxuICAgICAgICAgICAgbGVmdDogW1xuICAgICAgICAgICAgICBkbHEubWV0cmljQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzVmlzaWJsZSh7IGxhYmVsOiAnVmlzaWJsZSBNZXNzYWdlcycsIHBlcmlvZDogRHVyYXRpb24ubWludXRlcygxKSB9KSxcbiAgICAgICAgICAgICAgZGxxLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc05vdFZpc2libGUoeyBsYWJlbDogJ0hpZGRlbiBNZXNzYWdlcycsIHBlcmlvZDogRHVyYXRpb24ubWludXRlcygxKSB9KSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBsZWZ0WUF4aXM6IHsgbWluOiAwIH0sXG4gICAgICAgICAgICByaWdodDogW1xuICAgICAgICAgICAgICBkbHEubWV0cmljQXBwcm94aW1hdGVBZ2VPZk9sZGVzdE1lc3NhZ2UoeyBsYWJlbDogJ09sZGVzdCBNZXNzYWdlIEFnZScsIHBlcmlvZDogRHVyYXRpb24ubWludXRlcygxKSB9KSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICByaWdodFlBeGlzOiB7IG1pbjogMCB9LFxuICAgICAgICAgIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IEdyYXBoV2lkZ2V0KHtcbiAgICAgICAgICAgIGhlaWdodDogNixcbiAgICAgICAgICAgIHdpZHRoOiAxMixcbiAgICAgICAgICAgIHRpdGxlOiAnUXVhbGl0eSBNZXRyaWNzJyxcbiAgICAgICAgICAgIGxlZnQ6IFtcbiAgICAgICAgICAgICAgZmlsbE1ldHJpYyh0aGlzLm1ldHJpY05vdEpzaWlFbmFibGVkQ291bnQoeyBsYWJlbDogJ05vdCBhIGpzaWkgcGFja2FnZScgfSksIDApLFxuICAgICAgICAgICAgICBmaWxsTWV0cmljKHRoaXMubWV0cmljSW5lbGlnaWJsZUxpY2Vuc2UoeyBsYWJlbDogJ0luZWxpZ2libGUgTGljZW5zZScgfSksIDApLFxuICAgICAgICAgICAgICBmaWxsTWV0cmljKHRoaXMubWV0cmljRGVueUxpc3RlZENvdW50KHsgbGFiZWw6ICdEZW55IExpc3RlZCcgfSksIDApLFxuICAgICAgICAgICAgICBmaWxsTWV0cmljKHRoaXMubWV0cmljRGVsZXRlZENvdW50KHsgbGFiZWw6ICdEZWxldGlvbiBFdmVudHMnIH0pLCAwKSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBsZWZ0WUF4aXM6IHsgbWluOiAwIH0sXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH07XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY0RlbnlMaXN0ZWRDb3VudChvcHRzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBNZXRyaWMoe1xuICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDUpLFxuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuU1VNLFxuICAgICAgLi4ub3B0cyxcbiAgICAgIGRpbWVuc2lvbnNNYXA6IHtcbiAgICAgICAgW0RPTUFJTl9PV05FUl9ESU1FTlNJT05dOiB0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ckRvbWFpbk93bmVyLFxuICAgICAgICBbRE9NQUlOX05BTUVfRElNRU5TSU9OXTogdGhpcy5wcm9wcy5yZXBvc2l0b3J5LmF0dHJEb21haW5OYW1lLFxuICAgICAgICBbUkVQT1NJVE9SWV9OQU1FX0RJTUVOU0lPTl06IHRoaXMucHJvcHMucmVwb3NpdG9yeS5hdHRyTmFtZSxcbiAgICAgIH0sXG4gICAgICBtZXRyaWNOYW1lOiBNZXRyaWNOYW1lLkRFTllfTElTVEVEX0NPVU5ULFxuICAgICAgbmFtZXNwYWNlOiBNRVRSSUNTX05BTUVTUEFDRSxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY0luZWxpZ2libGVMaWNlbnNlKG9wdHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5TVU0sXG4gICAgICAuLi5vcHRzLFxuICAgICAgZGltZW5zaW9uc01hcDoge1xuICAgICAgICBbRE9NQUlOX09XTkVSX0RJTUVOU0lPTl06IHRoaXMucHJvcHMucmVwb3NpdG9yeS5hdHRyRG9tYWluT3duZXIgPz8gQXdzLkFDQ09VTlRfSUQsXG4gICAgICAgIFtET01BSU5fTkFNRV9ESU1FTlNJT05dOiB0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ckRvbWFpbk5hbWUsXG4gICAgICAgIFtSRVBPU0lUT1JZX05BTUVfRElNRU5TSU9OXTogdGhpcy5wcm9wcy5yZXBvc2l0b3J5LmF0dHJOYW1lLFxuICAgICAgfSxcbiAgICAgIG1ldHJpY05hbWU6IE1ldHJpY05hbWUuSU5FTElHSUJMRV9MSUNFTlNFLFxuICAgICAgbmFtZXNwYWNlOiBNRVRSSUNTX05BTUVTUEFDRSxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBtZXRyaWNEZWxldGVkQ291bnQob3B0cz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLlNVTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBkaW1lbnNpb25zTWFwOiB7XG4gICAgICAgIFtET01BSU5fT1dORVJfRElNRU5TSU9OXTogdGhpcy5wcm9wcy5yZXBvc2l0b3J5LmF0dHJEb21haW5Pd25lciA/PyBBd3MuQUNDT1VOVF9JRCxcbiAgICAgICAgW0RPTUFJTl9OQU1FX0RJTUVOU0lPTl06IHRoaXMucHJvcHMucmVwb3NpdG9yeS5hdHRyRG9tYWluTmFtZSxcbiAgICAgICAgW1JFUE9TSVRPUllfTkFNRV9ESU1FTlNJT05dOiB0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ck5hbWUsXG4gICAgICB9LFxuICAgICAgbWV0cmljTmFtZTogTWV0cmljTmFtZS5ERUxFVEVEX0NPVU5ULFxuICAgICAgbmFtZXNwYWNlOiBNRVRSSUNTX05BTUVTUEFDRSxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIG1ldHJpY05vdEpzaWlFbmFibGVkQ291bnQob3B0cz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLlNVTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBkaW1lbnNpb25zTWFwOiB7XG4gICAgICAgIFtET01BSU5fT1dORVJfRElNRU5TSU9OXTogdGhpcy5wcm9wcy5yZXBvc2l0b3J5LmF0dHJEb21haW5Pd25lciA/PyBBd3MuQUNDT1VOVF9JRCxcbiAgICAgICAgW0RPTUFJTl9OQU1FX0RJTUVOU0lPTl06IHRoaXMucHJvcHMucmVwb3NpdG9yeS5hdHRyRG9tYWluTmFtZSxcbiAgICAgICAgW1JFUE9TSVRPUllfTkFNRV9ESU1FTlNJT05dOiB0aGlzLnByb3BzLnJlcG9zaXRvcnkuYXR0ck5hbWUsXG4gICAgICB9LFxuICAgICAgbWV0cmljTmFtZTogTWV0cmljTmFtZS5OT1RfSlNJSV9FTkFCTEVEX0NPVU5ULFxuICAgICAgbmFtZXNwYWNlOiBNRVRSSUNTX05BTUVTUEFDRSxcbiAgICB9KTtcbiAgfVxufVxuIl19