"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("../backend/shared/constants");
const caching_1 = require("../caching");
const monitored_certificate_1 = require("../monitored-certificate");
const storage_1 = require("../s3/storage");
const temp_file_1 = require("../temp-file");
const config_1 = require("./config");
const response_function_1 = require("./response-function");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this);
        this.bucket = storageFactory.newBucket(this, 'WebsiteBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        // generate a stable unique id for the cloudfront function and use it
        // both for the function name and the logical id of the function so if
        // it is changed the function will be recreated.
        // see https://github.com/aws/aws-cdk/issues/15523
        const functionId = `AddHeadersFunction${this.node.addr}`;
        const behaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_OPTIMIZED,
            functionAssociations: [{
                    function: new response_function_1.ResponseFunction(this, functionId, {
                        functionName: functionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket), ...behaviorOptions },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        // The base URL is currently the custom DNS if any was used, or the distribution domain name.
        // This needs changing in case, for example, we add support for a custom URL prefix.
        this.baseUrl = `https://${props.domain ? props.domain.zone.zoneName : this.distribution.distributionDomainName}`;
        const jsiiObjOrigin = new origins.S3Origin(props.packageData);
        this.distribution.addBehavior('/data/*', jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.CATALOG_KEY}`, jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.VERSION_TRACKER_KEY}`, jsiiObjOrigin, behaviorOptions);
        if (props.packageStats) {
            this.distribution.addBehavior(`/${props.packageStats.statsKey}`, jsiiObjOrigin, behaviorOptions);
        }
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // Monitor certificate expiration
            if ((_a = props.domain.monitorCertificateExpiration) !== null && _a !== void 0 ? _a : true) {
                const monitored = new monitored_certificate_1.MonitoredCertificate(this, 'ExpirationMonitor', {
                    certificate: props.domain.cert,
                    domainName: props.domain.zone.zoneName,
                });
                props.monitoring.addHighSeverityAlarm('ACM Certificate Expiry', monitored.alarmAcmCertificateExpiresSoon);
                props.monitoring.addHighSeverityAlarm('Endpoint Certificate Expiry', monitored.alarmEndpointCertificateExpiresSoon);
            }
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            sources: [s3deploy.Source.asset(webappDir)],
            cacheControl: caching_1.CacheStrategy.default().toArray(),
        });
        // Generate config.json to customize frontend behavior
        const config = new config_1.WebappConfig({
            packageLinks: props.packageLinks,
            packageTags: props.packageTags,
            packageTagGroups: props.packageTagGroups,
            featuredPackages: props.featuredPackages,
            showPackageStats: (_b = props.showPackageStats) !== null && _b !== void 0 ? _b : props.packageStats !== undefined,
            featureFlags: props.featureFlags,
            categories: props.categories,
        });
        // Generate preload.js
        const preloadScript = new temp_file_1.TempFile('preload.js', (_c = props.preloadScript) !== null && _c !== void 0 ? _c : '');
        new s3deploy.BucketDeployment(this, 'DeployWebsiteConfig', {
            sources: [s3deploy.Source.asset(config.file.dir), s3deploy.Source.asset(preloadScript.dir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            cacheControl: caching_1.CacheStrategy.default().toArray(),
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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