"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const CONFIG_BUCKET_NAME = env_lambda_shared_1.requireEnv('CONFIG_BUCKET_NAME');
    const CONFIG_FILE_KEY = env_lambda_shared_1.requireEnv('CONFIG_FILE_KEY');
    // Load configuration
    const { packageTags: packageTagsConfig, packageLinks: allowedLinks } = await getConfig(CONFIG_BUCKET_NAME, CONFIG_FILE_KEY);
    const codeArtifactProps = (function () {
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            return undefined;
        }
        const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
        const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
        const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
        return { endpoint, domain, domainOwner, apiEndpoint };
    })();
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const { integrity: integrityCheck } = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let parsedAssembly;
        let constructFramework;
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            parsedAssembly = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            // "types" is huge and not needed downstream because doc generation happens in the backend,
            // so we drop it. See https://github.com/cdklabs/construct-hub-webapp/issues/691
            delete parsedAssembly.types;
            constructFramework = detectConstructFramework(parsedAssembly);
            const { license, name, version } = parsedAssembly;
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, packageJsonObj)) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        if (codeArtifactProps) {
            console.log('Publishing to the internal CodeArtifact...');
            try {
                const { publishConfig } = packageJsonObj;
                if (publishConfig) {
                    console.log('Not publishing to CodeArtifact due to the presence of publishConfig in package.json: ', publishConfig);
                }
                else {
                    await code_artifact_lambda_shared_1.codeArtifactPublishPackage(Buffer.from(tarball.Body), codeArtifactProps);
                }
            }
            catch (err) {
                console.error('Failed publishing to CodeArtifact: ', err);
            }
        }
        const metadata = {
            constructFramework,
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public, max-age=86400, must-revalidate, s-maxage=300, proxy-revalidate',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: Buffer.from(JSON.stringify(parsedAssembly), 'utf-8'),
            CacheControl: 'public, max-age: 86400, must-revalidate, s-maxage=300, proxy-revalidate',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Determines the Construct framework used by the provided assembly.
 *
 * @param assembly the assembly for which a construct framework should be
 *                 identified.
 *
 * @returns a construct framework if one could be identified.
 */
function detectConstructFramework(assembly) {
    var _a;
    let name;
    let nameAmbiguous = false;
    let majorVersion;
    let majorVersionAmbiguous = false;
    detectConstructFrameworkPackage(assembly.name, assembly.version);
    for (const depName of Object.keys((_a = assembly.dependencyClosure) !== null && _a !== void 0 ? _a : {})) {
        detectConstructFrameworkPackage(depName);
        if (nameAmbiguous) {
            return undefined;
        }
    }
    return name && { name, majorVersion: majorVersionAmbiguous ? undefined : majorVersion };
    function detectConstructFrameworkPackage(packageName, versionRange) {
        var _a, _b;
        if (versionRange === void 0) { versionRange = (_a = assembly.dependencies) === null || _a === void 0 ? void 0 : _a[packageName]; }
        if (packageName.startsWith('@aws-cdk/') || packageName === 'aws-cdk-lib' || packageName === 'monocdk') {
            if (name && name !== "aws-cdk" /* AWS_CDK */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "aws-cdk" /* AWS_CDK */;
        }
        else if (packageName === 'cdktf' || packageName.startsWith('@cdktf/')) {
            if (name && name !== "cdktf" /* CDKTF */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdktf" /* CDKTF */;
        }
        else if (packageName === 'cdk8s' || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(packageName)) {
            if (name && name !== "cdk8s" /* CDK8S */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdk8s" /* CDK8S */;
        }
        else {
            return;
        }
        if (versionRange) {
            const major = (_b = semver_1.minVersion(versionRange)) === null || _b === void 0 ? void 0 : _b.major;
            if (majorVersion != null && majorVersion !== major) {
                // Identified multiple candidates, so this is ambiguous...
                majorVersionAmbiguous = true;
            }
            majorVersion = major;
        }
        return;
    }
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
/**
 * Looks for the ingestion configuration file in the passed bucket and parses
 * it. If it is not found or invalid then a default is returned.
 */
async function getConfig(bucket, key) {
    var _a;
    const defaultConfig = {
        packageTags: [],
        packageLinks: [],
    };
    try {
        const req = await aws.s3().getObject({
            Bucket: bucket,
            Key: key,
        }).promise();
        const body = (_a = req === null || req === void 0 ? void 0 : req.Body) === null || _a === void 0 ? void 0 : _a.toString();
        if (body) {
            return JSON.parse(body);
        }
        return defaultConfig;
    }
    catch (e) {
        console.error(e);
        return defaultConfig;
    }
}
//# sourceMappingURL=data:application/json;base64,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