import { Metric, MetricOptions } from '@aws-cdk/aws-cloudwatch';
import { IFunction } from '@aws-cdk/aws-lambda';
import { RetentionDays } from '@aws-cdk/aws-logs';
import { IBucket } from '@aws-cdk/aws-s3';
import { Construct, Duration } from '@aws-cdk/core';
import { Monitoring } from '../../monitoring';
import { DocumentationLanguage } from '../shared/language';
export interface InventoryProps {
    /**
     * The data storage bucket.
     */
    readonly bucket: IBucket;
    /**
     * The `Monitoring` instance to use for reporting this canary's health.
     */
    readonly monitoring: Monitoring;
    /**
     * How long should canary logs be retained?
     */
    readonly logRetention?: RetentionDays;
    /**
     * The rate at which the canary should run.
     *
     * @default Duration.minutes(5)
     */
    readonly scheduleRate?: Duration;
}
/**
 * Periodically computes an inventory of all indexed packages into the storage
 * bucket, and produces metrics with an overview of the index' state.
 */
export declare class Inventory extends Construct {
    private readonly canary;
    constructor(scope: Construct, id: string, props: InventoryProps);
    get function(): IFunction;
    metricMissingPackageMetadataCount(opts?: MetricOptions): Metric;
    metricMissingAssemblyCount(opts?: MetricOptions): Metric;
    metricPackageCount(opts?: MetricOptions): Metric;
    metricPackageMajorCount(opts?: MetricOptions): Metric;
    metricPackageVersionCount(opts?: MetricOptions): Metric;
    metricMissingPackageTarballCount(opts?: MetricOptions): Metric;
    metricUninstallablePackageCount(opts?: MetricOptions): Metric;
    metricSubmoduleCount(opts?: MetricOptions): Metric;
    metricUnknownObjectCount(opts?: MetricOptions): Metric;
    /**
     * The count of packages for which all versions are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package major versions for which all versions are missing a
     * documnetation artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingMajorVersionCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package versions that are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageVersionCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package version submodules that are missing a documnetation
     * artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingSubmoduleCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of packages that have at least one version for which there is
     * available documentation in the provided `DocumentationLanguage`.
     */
    metricSupportedPackageCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package major versions that have at least one version for
     * which there is available documentation in the provided
     * `DocumentationLanguage`.
     */
    metricSupportedMajorVersionCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package versions that have available documentation in the
     * provided `DocumentationLanguage`.
     */
    metricSupportedPackageVersionCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package version submodules that have available documentation
     * in the provided `DocumentationLanguage`.
     */
    metricSupportedSubmoduleCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of packages that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package major versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedMajorVersionCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageVersionCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package version submodules that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedSubmoduleCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of packages that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyPackageCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package major versions that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyMajorVersionCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package versions that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyPackageVersionCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
    /**
     * The count of package version submodules that have a language specific corrupt assembly.
     */
    metricCorruptAssemblySubmoduleCount(language: DocumentationLanguage, opts?: MetricOptions): Metric;
}
