"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        const repository = ((_a = props.vpcEndpoints) === null || _a === void 0 ? void 0 : _a.codeArtifact) && props.vpcEndpoints.codeArtifactApi
            ? (_b = props.codeArtifact) === null || _b === void 0 ? void 0 : _b.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
            // Set embedded metrics format environment to "Local", to have a consistent experience.
            AWS_EMF_ENVIRONMENT: 'Local',
        };
        if ((_c = props.vpcEndpoints) === null || _c === void 0 ? void 0 : _c.codeArtifactApi) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        this.logGroup = new aws_logs_1.LogGroup(this, 'LogGroup', { retention: props.logRetention });
        this.containerDefinition = new transliterator_1.Transliterator(this, 'Resource', {
            environment,
            logging: aws_ecs_1.LogDrivers.awsLogs({ logGroup: this.logGroup, streamPrefix: 'transliterator' }),
            taskDefinition: new aws_ecs_1.FargateTaskDefinition(this, 'TaskDefinition', {
                cpu: 4096,
                memoryLimitMiB: 8192,
            }),
        });
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.taskDefinition.taskRole);
        // The task handler reads & writes to this bucket.
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.PACKAGE_KEY_SUFFIX}`);
        bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.UNINSTALLABLE_PACKAGE_SUFFIX}`);
        bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.UNINSTALLABLE_PACKAGE_SUFFIX}`);
        for (const language of language_1.DocumentationLanguage.ALL) {
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.NOT_SUPPORTED_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
        }
        const executionRole = this.taskDefinition.obtainExecutionRole();
        (_d = props.vpcEndpoints) === null || _d === void 0 ? void 0 : _d.ecrApi.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:GetAuthorizationToken',
            ],
            resources: ['*'],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_e = props.vpcEndpoints) === null || _e === void 0 ? void 0 : _e.ecr.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchGetImage',
            ],
            // We cannot get the ECR repository info from an asset... So scoping down to same-account repositories instead...
            resources: [core_1.Stack.of(this).formatArn({ service: 'ecr', resource: 'repository', arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME, resourceName: '*' })],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_f = props.vpcEndpoints) === null || _f === void 0 ? void 0 : _f.cloudWatchLogs.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: this.logGroup.logGroupName }),
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: `${this.logGroup.logGroupName}:log-stream:*` }),
            ],
            principals: [executionRole],
            sid: 'Allow-Logging',
        }));
        (_g = props.vpcEndpoints) === null || _g === void 0 ? void 0 : _g.stepFunctions.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
                'states:SendTaskSuccess',
            ],
            resources: ['*'],
            principals: [this.taskDefinition.taskRole],
            sid: 'Allow-StepFunctions-Callbacks',
        }));
    }
    get taskDefinition() {
        return this.containerDefinition.taskDefinition;
    }
    createEcsRunTask(scope, id, opts) {
        return new aws_stepfunctions_tasks_1.EcsRunTask(scope, id, {
            // The container sends heartbeats every minute, but when the runloop will
            // actually get to submitting it is fairly dependent on the async
            // workload's nature... so we don't rely on it all too strongly, and
            // default to a 5 minutes timeout here as a minimal protection. Options
            // may override this liberally if they know better.
            heartbeat: core_1.Duration.minutes(5),
            ...opts,
            containerOverrides: [{
                    containerDefinition: this.containerDefinition,
                    command: aws_stepfunctions_1.JsonPath.listAt('$'),
                    environment: [
                        { name: 'SFN_TASK_TOKEN', value: aws_stepfunctions_1.JsonPath.taskToken },
                    ],
                }],
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            launchTarget: new aws_stepfunctions_tasks_1.EcsFargateLaunchTarget({ platformVersion: aws_ecs_1.FargatePlatformVersion.VERSION1_4 }),
            subnets: opts.vpcSubnets,
            securityGroups: opts.securityGroups,
            taskDefinition: this.taskDefinition,
        });
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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