"use strict";
var __classPrivateFieldGet = (this && this.__classPrivateFieldGet) || function (receiver, privateMap) {
    if (!privateMap.has(receiver)) {
        throw new TypeError("attempted to get private field on non-instance");
    }
    return privateMap.get(receiver);
};
var __classPrivateFieldSet = (this && this.__classPrivateFieldSet) || function (receiver, privateMap, value) {
    if (!privateMap.has(receiver)) {
        throw new TypeError("attempted to set private field on non-instance");
    }
    privateMap.set(receiver, value);
    return value;
};
var _catalog;
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const https = require("https");
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const JSONStream = require("JSONStream");
const aws = require("../../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../../backend/shared/env.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
/**
 * This package canary monitors the availability of the versions of a specified
 * package in the ConstructHub catalog. It publishes metrics that help
 * understand how much time passes between a pakcage appearing in the public
 * registry and it's availability in the ConstructHub instance.
 *
 * From the moment a package has been published, and until it appeared in
 * catalog, the `MetricName.DWELL_TIME` metric is emitted.
 *
 * Once the package has appeared in catalog, and until a new package version is
 * identified in npmjs.com, the `MetricName.TIME_TO_CATALOG` metric is emitted.
 *
 * If a new package version is published before the previous one has appeared
 * in catalog, both versions will be tracked at the same time, and the metrics
 * will receive one sample per tracked version.
 */
async function handler(event) {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const packageName = env_lambda_shared_1.requireEnv("PACKAGE_NAME" /* PACKAGE_NAME */);
    const stateBucket = env_lambda_shared_1.requireEnv("PACKAGE_CANARY_BUCKET_NAME" /* PACKAGE_CANARY_BUCKET_NAME */);
    const constructHubEndpoint = env_lambda_shared_1.requireEnv("CONSTRUCT_HUB_BASE_URL" /* CONSTRUCT_HUB_BASE_URL */);
    const stateService = new CanaryStateService(stateBucket);
    const constructHub = new ConstructHub(constructHubEndpoint);
    const latest = await stateService.latest(packageName);
    const state = (_a = await stateService.load(packageName)) !== null && _a !== void 0 ? _a : {
        latest: {
            ...latest,
            // If that latest version is ALREADY in catalog, pretend it was
            // "instantaneously" there, so we avoid possibly reporting an breach of
            // SLA alarm, when we really just observed presence of the package in
            // catalog too late, for example on first deployment of the canary.
            availableAt: await constructHub.isInCatalog(packageName, latest.version)
                ? latest.publishedAt
                : undefined,
        },
        pending: {},
    };
    console.log(`Initial state: ${JSON.stringify(state, null, 2)}`);
    // If the current "latest" isn't the one from state, it needs updating.
    updateLatestIfNeeded(state, latest);
    try {
        await aws_embedded_metrics_1.metricScope((metrics) => () => {
            // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
            metrics.setDimensions();
            metrics.putMetric("TrackedVersionCount" /* TRACKED_VERSION_COUNT */, Object.keys(state.pending).length + 1, aws_embedded_metrics_1.Unit.Count);
        })();
        for (const versionState of [state.latest, ...Object.values((_b = state.pending) !== null && _b !== void 0 ? _b : {})]) {
            console.log(`Checking state of ${versionState.version}, current: ${JSON.stringify(versionState, null, 2)}`);
            await aws_embedded_metrics_1.metricScope((metrics) => async () => {
                // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
                metrics.setDimensions();
                metrics.setProperty('PackageName', packageName);
                metrics.setProperty('PackageVersion', versionState.version);
                metrics.setProperty('IsLatest', state.latest.version === versionState.version);
                if (!versionState.availableAt) {
                    if (versionState.version === state.latest.version) {
                        if (await constructHub.isInCatalog(packageName, versionState.version)) {
                            versionState.availableAt = new Date();
                        }
                    }
                    else {
                        // Non-current versions will probably never make it to catalog (they're older than the
                        // current version), so instead, we check whether they have TypeScript documentation.
                        if (await constructHub.hasTypeScriptDocumentation(packageName, versionState.version)) {
                            versionState.availableAt = new Date();
                        }
                    }
                }
                if (versionState.availableAt) {
                    // Tells us how long it's taken for the package to make it to catalog after it was published.
                    metrics.putMetric("TimeToCatalog" /* TIME_TO_CATALOG */, (versionState.availableAt.getTime() - versionState.publishedAt.getTime()) / 1000, aws_embedded_metrics_1.Unit.Seconds);
                    // Stop tracking that version, as it's now available.
                    if (versionState.version in state.pending) {
                        delete state.pending[versionState.version];
                    }
                }
                else {
                    // Tells us how long we've been waiting for this version to show up, so far.
                    metrics.putMetric("DwellTime" /* DWELL_TIME */, (Date.now() - versionState.publishedAt.getTime()) / 1000, aws_embedded_metrics_1.Unit.Seconds);
                }
            })();
        }
    }
    finally {
        await stateService.save(packageName, state);
    }
}
exports.handler = handler;
class ConstructHub {
    constructor(baseUrl) {
        this.baseUrl = baseUrl;
        _catalog.set(this, void 0);
    }
    /**
     * Determines whether the specified package version is present in the catalog
     * object or not.
     *
     * @param packageName    the name of the checked package.
     * @param packageVersion the version of the checked package.
     *
     * @returns `true` IIF the exact package version is found in the catalog.
     */
    async isInCatalog(packageName, packageVersion) {
        const catalog = await this.getCatalog();
        const filtered = catalog.packages.filter((p) => p.name === packageName && p.version === packageVersion);
        if (filtered.length > 1) {
            throw new Error(`Found multiple entries for ${packageName}@${packageVersion} in catalog`);
        }
        return filtered.length === 1;
    }
    /**
     * Checks whether TypeScript documentation exists in ConstructHub for the
     * specified package version.
     *
     * @param packageName    the name of the checked package.
     * @param packageVersion the version of the checked package.
     *
     * @returns `true` IIF the `docs-typescript.md` document exists for the
     *          specified package.
     */
    async hasTypeScriptDocumentation(packageName, packageVersion) {
        return new Promise((ok, ko) => {
            const url = `${this.baseUrl}/data/${packageName}/v${packageVersion}/docs-typescript.md`;
            https.request(url, { method: 'HEAD' }, (res) => {
                var _a;
                if (res.statusCode === 200) {
                    // This returns HTTP 200 with text/html if it's a 404, due to how
                    // we configured CloudFront behaviors.
                    return ok(!!((_a = res.headers['content-type']) === null || _a === void 0 ? void 0 : _a.startsWith('text/markdown')));
                }
                const err = new Error(`HEAD ${url} -- HTTP ${res.statusCode} (${res.statusMessage})`);
                Error.captureStackTrace(err);
                ko(err);
            }).end();
        });
    }
    async getCatalog() {
        if (__classPrivateFieldGet(this, _catalog)) {
            return __classPrivateFieldGet(this, _catalog);
        }
        return __classPrivateFieldSet(this, _catalog, await getJSON(`${this.baseUrl}/catalog.json`));
    }
}
_catalog = new WeakMap();
class CanaryStateService {
    constructor(bucketName) {
        this.bucketName = bucketName;
    }
    /**
     * Save the state to the bucket.
     */
    async save(packageName, state) {
        const url = this.url(packageName);
        console.log(`Saving to ${url}: ${JSON.stringify(state, null, 2)}`);
        await aws.s3().putObject({
            Bucket: this.bucketName,
            Key: this.key(packageName),
            Body: JSON.stringify(state, null, 2),
            ContentType: 'application/json',
        }).promise();
    }
    /**
     * Load the state file for this package from the bucket.
     */
    async load(packageName) {
        console.log(`Loading state for package '${packageName}'`);
        const objectKey = this.key(packageName);
        const url = this.url(packageName);
        console.log(`Fetching: ${url}`);
        const data = await aws.s3().getObject({ Bucket: this.bucketName, Key: objectKey }).promise()
            .catch((err) => err.code !== 'NoSuchKey'
            ? Promise.reject(err)
            : Promise.resolve({ /* no data */}));
        if (!(data === null || data === void 0 ? void 0 : data.Body)) {
            console.log(`Not found: ${url}`);
            return undefined;
        }
        console.log(`Loaded: ${url}`);
        return JSON.parse(data.Body.toString('utf-8'), (key, value) => {
            if (key === 'publishedAt' || key === 'availableAt') {
                return new Date(value);
            }
            return value;
        });
    }
    /**
     * Create a state from the latest version of the package.
     */
    async latest(packageName) {
        console.log(`Fetching latest version information from NPM: ${packageName}`);
        const version = await getJSON(`https://registry.npmjs.org/${encodeURIComponent(packageName)}/latest`, ['version']);
        const publishedAt = await getJSON(`https://registry.npmjs.org/${encodeURIComponent(packageName)}`, ['time', version]);
        console.log(`Package: ${packageName} | Version : ${version} | Published At: ${publishedAt}`);
        return { version, publishedAt: new Date(publishedAt) };
    }
    key(packageName) {
        return `${"package-canary/" /* STATE_PREFIX */}${packageName}${".state.json" /* STATE_SUFFIX */}`;
    }
    url(packageName) {
        return `s3://${this.bucketName}/${this.key(packageName)}`;
    }
}
/**
 * Makes a request to the provided URL and returns the response after having
 * parsed it from JSON.
 *
 * @param url the URL to get.
 * @param jsonPath a JSON path to extract only a subset of the object.
 */
function getJSON(url, jsonPath) {
    return new Promise((ok, ko) => {
        https.get(url, { headers: { 'Accept': 'application/json', 'Accept-Encoding': 'identity' } }, (res) => {
            if (res.statusCode !== 200) {
                const error = new Error(`GET ${url} - HTTP ${res.statusCode} (${res.statusMessage})`);
                Error.captureStackTrace(error);
                return ko(error);
            }
            res.once('error', ko);
            const json = JSONStream.parse(jsonPath);
            json.once('data', ok);
            json.once('error', ko);
            const plainPayload = res.headers['content-encoding'] === 'gzip' ? gunzip(res) : res;
            plainPayload.pipe(json, { end: true });
        });
    });
}
/**
 * Updates the `latest` property of `state` ti the provided `latest` value,
 * unless this is already the current latest.
 *
 * If the previous latest version does not have the `availableAt` property, adds
 * that to the `pending` set.
 *
 * @param state  the state to be updated.
 * @param latest the current "latest" version of the tracked package.
 */
function updateLatestIfNeeded(state, latest) {
    if (state.latest.version === latest.version) {
        return;
    }
    // If the current "latest" isn't available yet, add it to the `pending` versions.
    if (state.latest.availableAt == null) {
        // The TypeScript version of jsii doesn't do control flow analysis well enough here to
        // determine that the`if` branch guarantees `availableAt` is undefined here.
        state.pending[state.latest.version] = { ...state.latest, availableAt: undefined };
    }
    state.latest = latest;
}
function gunzip(readable) {
    const gz = zlib_1.createGunzip();
    readable.pipe(gz, { end: true });
    return gz;
}
//# sourceMappingURL=data:application/json;base64,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