"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const codepipeline = require("../lib");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
module.exports = {
    'Pipeline': {
        'can be passed an IAM role during pipeline creation'(test) {
            const stack = new cdk.Stack();
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com')
            });
            new codepipeline.Pipeline(stack, 'Pipeline', {
                role
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'RoleArn': {
                    'Fn::GetAtt': [
                        'Role1ABCC5F0',
                        'Arn',
                    ]
                }
            }));
            test.done();
        },
        'can be imported by ARN'(test) {
            const stack = new cdk.Stack();
            const pipeline = codepipeline.Pipeline.fromPipelineArn(stack, 'Pipeline', 'arn:aws:codepipeline:us-east-1:123456789012:MyPipeline');
            test.equal(pipeline.pipelineArn, 'arn:aws:codepipeline:us-east-1:123456789012:MyPipeline');
            test.equal(pipeline.pipelineName, 'MyPipeline');
            test.done();
        },
        'that is cross-region': {
            'validates that source actions are in the same account as the pipeline'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack', { env: { region: 'us-west-1', account: '123456789012' } });
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
                const sourceStage = pipeline.addStage({
                    stageName: 'Source',
                });
                const sourceAction = new fake_source_action_1.FakeSourceAction({
                    actionName: 'FakeSource',
                    output: new codepipeline.Artifact(),
                    region: 'ap-southeast-1',
                });
                test.throws(() => {
                    sourceStage.addAction(sourceAction);
                }, /Source action 'FakeSource' must be in the same region as the pipeline/);
                test.done();
            },
            'allows passing an Alias in place of the KMS Key in the replication Bucket'(test) {
                const app = new cdk.App();
                const replicationRegion = 'us-west-1';
                const replicationStack = new cdk.Stack(app, 'ReplicationStack', {
                    env: { region: replicationRegion, account: '123456789012' },
                });
                const replicationKey = new kms.Key(replicationStack, 'ReplicationKey');
                const replicationAlias = replicationKey.addAlias('alias/my-replication-alias');
                const replicationBucket = new s3.Bucket(replicationStack, 'ReplicationBucket', {
                    encryptionKey: replicationAlias,
                    bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(app, 'PipelineStack', {
                    env: { region: pipelineRegion, account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    crossRegionReplicationBuckets: {
                        [replicationRegion]: replicationBucket,
                    },
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: replicationRegion,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':kms:us-west-1:123456789012:alias/my-replication-alias',
                                            ],
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                }));
                assert_1.expect(replicationStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
                    'KeyPolicy': {
                        'Statement': [
                            {
                            // owning account management permissions - we don't care about them in this test
                            },
                            {
                                // KMS verifies whether the principal given in its key policy exists when creating that key.
                                // Since the replication bucket must be deployed before the pipeline,
                                // we cannot put the pipeline role as the principal here -
                                // hence, we put the account itself
                                'Action': [
                                    'kms:Decrypt',
                                    'kms:DescribeKey',
                                    'kms:Encrypt',
                                    'kms:ReEncrypt*',
                                    'kms:GenerateDataKey*',
                                ],
                                'Effect': 'Allow',
                                'Principal': {
                                    'AWS': {
                                        'Fn::Join': ['', [
                                                'arn:',
                                                { 'Ref': 'AWS::Partition' },
                                                ':iam::123456789012:root',
                                            ]],
                                    },
                                },
                                'Resource': '*',
                            },
                        ],
                    },
                }));
                test.done();
            },
            "generates ArtifactStores with the alias' ARN as the KeyID"(test) {
                const app = new cdk.App();
                const replicationRegion = 'us-west-1';
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(app, 'MyStack', {
                    env: { region: pipelineRegion, account: '123456789012' },
                });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                    region: replicationRegion,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': {
                                        'Fn::Join': [
                                            '',
                                            [
                                                'arn:',
                                                {
                                                    'Ref': 'AWS::Partition',
                                                },
                                                ':kms:us-west-1:123456789012:alias/s-west-1tencryptionalias9b344b2b8e6825cb1f7d',
                                            ],
                                        ],
                                    },
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                }));
                assert_1.expect(pipeline.crossRegionSupport[replicationRegion].stack).to(assert_1.haveResourceLike('AWS::KMS::Alias', {
                    'DeletionPolicy': 'Delete',
                    'UpdateReplacePolicy': 'Delete',
                }, assert_1.ResourcePart.CompleteDefinition));
                test.done();
            },
            'allows passing an imported Bucket and Key for the replication Bucket'(test) {
                const replicationRegion = 'us-west-1';
                const pipelineRegion = 'us-west-2';
                const pipelineStack = new cdk.Stack(undefined, undefined, {
                    env: { region: pipelineRegion },
                });
                const sourceOutput = new codepipeline.Artifact();
                new codepipeline.Pipeline(pipelineStack, 'Pipeline', {
                    crossRegionReplicationBuckets: {
                        [replicationRegion]: s3.Bucket.fromBucketAttributes(pipelineStack, 'ReplicationBucket', {
                            bucketArn: 'arn:aws:s3:::my-us-west-1-replication-bucket',
                            encryptionKey: kms.Key.fromKeyArn(pipelineStack, 'ReplicationKey', `arn:aws:kms:${replicationRegion}:123456789012:key/1234-5678-9012`),
                        }),
                    },
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({
                                    actionName: 'Source',
                                    output: sourceOutput,
                                })],
                        },
                        {
                            stageName: 'Build',
                            actions: [new fake_build_action_1.FakeBuildAction({
                                    actionName: 'Build',
                                    input: sourceOutput,
                                })],
                        },
                    ],
                });
                assert_1.expect(pipelineStack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                    'ArtifactStores': [
                        {
                            'Region': replicationRegion,
                            'ArtifactStore': {
                                'Type': 'S3',
                                'Location': 'my-us-west-1-replication-bucket',
                                'EncryptionKey': {
                                    'Type': 'KMS',
                                    'Id': 'arn:aws:kms:us-west-1:123456789012:key/1234-5678-9012',
                                },
                            },
                        },
                        {
                            'Region': pipelineRegion,
                        },
                    ],
                }));
                test.done();
            },
        },
        'that is cross-account': {
            'does not allow passing a dynamic value in the Action account property'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack', { env: { account: '123456789012' } });
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Source', output: sourceOutput })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({ stageName: 'Build' });
                test.throws(() => {
                    buildStage.addAction(new fake_build_action_1.FakeBuildAction({
                        actionName: 'FakeBuild',
                        input: sourceOutput,
                        account: cdk.Aws.ACCOUNT_ID,
                    }));
                }, /The 'account' property must be a concrete value \(action: 'FakeBuild'\)/);
                test.done();
            },
            'does not allow an env-agnostic Pipeline Stack if an Action account has been provided'(test) {
                const app = new cdk.App();
                const stack = new cdk.Stack(app, 'PipelineStack');
                const sourceOutput = new codepipeline.Artifact();
                const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                    stages: [
                        {
                            stageName: 'Source',
                            actions: [new fake_source_action_1.FakeSourceAction({ actionName: 'Source', output: sourceOutput })],
                        },
                    ],
                });
                const buildStage = pipeline.addStage({ stageName: 'Build' });
                test.throws(() => {
                    buildStage.addAction(new fake_build_action_1.FakeBuildAction({
                        actionName: 'FakeBuild',
                        input: sourceOutput,
                        account: '123456789012',
                    }));
                }, /Pipeline stack which uses cross-environment actions must have an explicitly set account/);
                test.done();
            },
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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