"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        var _b, _c, _d;
        super({
            ...options,
            github: false,
            jest: false,
            package: false,
            prettier: true,
            projenrcTs: true,
            release: false,
            sampleCode: false,
            defaultReleaseBranch: "mainline",
        });
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.implicitDependencies = ((_b = this.nxConfig) === null || _b === void 0 ? void 0 : _b.implicitDependencies) || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(((_c = this.nxConfig) === null || _c === void 0 ? void 0 : _c.nxIgnore) || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(((_d = this.nxConfig) === null || _d === void 0 ? void 0 : _d.targetDependencies) || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.synthesizeNonNodePackageJson();
        super.synth();
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            var _b;
            // Disable default task on subprojects as this isn't supported in a monorepo
            (_b = subProject.defaultTask) === null || _b === void 0 ? void 0 : _b.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            manifest.name = subProject.name;
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        var _b;
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)),
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)),
                nohoist: (_b = this.workspaceConfig) === null || _b === void 0 ? void 0 : _b.noHoist,
            });
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "@aws-prototyping-sdk/nx-monorepo.NxMonorepoProject", version: "0.0.0" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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