from logging import Logger, getLogger
from typing import Any, Callable, Dict, Iterator, Optional, Type, Tuple, TypeVar, Union

import ipywidgets as widgets
import traitlets

from .types import TraitViewFactoryType
from .widgets import ModelViewWidget

default_logger = getLogger(__name__)


_trait_view_variant_factories: Dict[
    Type[traitlets.TraitType], TraitViewFactoryType
] = {}


def member_names_ordered(cls):
    return [
        k
        for c in reversed(cls.__mro__)
        for k, v in vars(c).items()
        if isinstance(v, traitlets.TraitType)
    ]


def request_constructor_for_variant(
    variant_kwarg_pairs, variant: Type[widgets.Widget] = None
) -> Tuple[Any, Dict[str, Any]]:
    """Return best widget constructor from a series of candidates. Attempt to satisfy variant.

    :param variant_kwarg_pairs: sequence of (widget_cls, kwarg) pairs
    :param variant: optional requested variant.
    :return:
    """
    assert variant_kwarg_pairs

    for cls, kwargs in variant_kwarg_pairs:
        if cls is variant:
            break
    else:
        default_logger.debug(
            f"Unable to find variant {variant} in {variant_kwarg_pairs}"
        )

    return cls, kwargs


def get_trait_view_variant_factory(
    trait_type: Type[traitlets.TraitType],
) -> TraitViewFactoryType:
    """Get a view factory for a given trait class

    :param trait_type: trait class
    :return:
    """
    for cls in trait_type.__mro__:
        try:
            return _trait_view_variant_factories[cls]
        except KeyError:
            continue
    raise ValueError(f"Couldn't find factory for {trait_type}")


def register_trait_view_variant_factory(
    *trait_types: Type[traitlets.TraitType], variant_factory: TraitViewFactoryType
):
    """Register a view factory for a given traitlet type(s)

    :param trait_types: trait class(es) to register
    :param variant_factory: view factory for trait class(es)
    :return:
    """
    for trait_type in trait_types:
        _trait_view_variant_factories[trait_type] = variant_factory


def unregister_trait_view_variant_factory(*trait_types: Type[traitlets.TraitType]):
    """Unregister a view factory for a given traitlet type(s)

    :param trait_types: trait class(es) to unregister
    :return:
    """
    for trait_type in trait_types:
        del _trait_view_variant_factories[trait_type]


def trait_view_variants(*trait_types: Type[traitlets.TraitType]):
    """Decorator for  registering view factory functions

    :param trait_types: trait class(es) to register
    :return:
    """

    def wrapper(factory):
        register_trait_view_variant_factory(*trait_types, variant_factory=factory)
        return factory

    return wrapper


class ViewFactoryContext:
    def __init__(self, factory: "ViewFactory", path: Tuple[str, ...]):
        self._factory = factory
        self.visited = set()
        self.path = path
        self.logger = factory.logger

    @property
    def name(self) -> Union[str, None]:
        if self.path:
            return self.path[-1]
        return None

    @property
    def display_name(self) -> Union[str, None]:
        if self.name is None:
            return None
        return self.name.replace("_", " ").title()

    def create_widgets_for_model_cls(self, model_cls: Type[traitlets.HasTraits]):
        return self._factory.create_widgets_for_model_cls(model_cls, self)

    def create_trait_view(self, trait: traitlets.TraitType, metadata: Dict[str, Any]):
        return self._factory.create_trait_view(trait, metadata, self)

    def resolve(self, name_or_cls: Union[type, str]) -> type:
        return self._factory.resolve(name_or_cls)

    def follow(self, name: str) -> "ViewFactoryContext":
        return type(self)(self._factory, self.path + (name,))


FilterType = Callable[
    [Type[traitlets.HasTraits], Tuple[str, ...], traitlets.TraitType], bool
]


TransformerType = Callable[
    [
        Type[traitlets.HasTraits],
        traitlets.TraitType,
        widgets.Widget,
        ViewFactoryContext,
    ],
    Optional[widgets.Widget],
]


VariantIterator = Iterator[Tuple[Type[widgets.Widget], Dict[str, Any]]]
TraitViewFactoryType = Callable[
    [traitlets.TraitType, Dict[str, Any], ViewFactoryContext], VariantIterator
]


T = TypeVar("T")


class ViewFactory:
    def __init__(
        self,
        logger: Logger = default_logger,
        namespace: Dict[str, Any] = None,
        filter_trait: FilterType = None,
    ):
        self.logger = logger
        self._namespace = namespace or {}
        self._visited = set()
        self._root_ctx = ViewFactoryContext(self, ())
        self._filter_trait = filter_trait

    def can_visit_trait(
        self, model_cls: Type[traitlets.HasTraits], trait, ctx: ViewFactoryContext
    ) -> bool:
        """Test to determine whether to visit a trait

        :param model_cls: model class object
        :param trait: trait instance
        :param ctx: factory context
        :return:
        """
        if callable(self._filter_trait):
            return self._filter_trait(model_cls, trait, ctx)
        return True

    def create_root_view(self, model: traitlets.HasTraits, metadata: Dict[str, Any] = None):
        """

        :param model:
        :param metadata: UI metadata
        :return:
        """
        model_view_cls = ModelViewWidget.specialise_for_cls(type(model))

        if metadata is None:
            metadata = {}

        return model_view_cls(ctx=self._root_ctx, value=model, **metadata)

    def create_trait_view(
        self,
        trait: traitlets.TraitType,
        metadata: Dict[str, Any] = None,
        ctx: ViewFactoryContext = None,
    ) -> widgets.Widget:
        """Return the best view constructor for a given trait

        :param trait: trait instance
        :param metadata: UI metadata
        :param ctx: factory context
        :return:
        """
        if ctx is None:
            ctx = self._root_ctx

        if metadata is None:
            metadata = {}

        factory = get_trait_view_variant_factory(type(trait))

        # Allow model or caller to set view metadata
        view_metadata = {**trait.metadata, **metadata}

        # Remove 'variant' field from metadata
        variant = view_metadata.pop("variant", None)

        # Allow library to propose variants
        supported_variants = list(factory(trait, view_metadata, ctx))
        if not supported_variants:
            raise ValueError(f"No variant found for {trait}")

        # Find variant
        if variant is None:
            cls, constructor_kwargs = supported_variants[-1]
        else:
            # Find the widget class for this variant, if possible
            cls, constructor_kwargs = request_constructor_for_variant(
                supported_variants, variant
            )

        # Traitlets cannot receive additional arguments.
        # Use factories should handle these themselves
        if issubclass(cls, traitlets.HasTraits):
            trait_names = set(cls.class_trait_names())
            constructor_kwargs = {
                k: v for k, v in constructor_kwargs.items() if k in trait_names
            }

        # Set widget disabled according to trait by default
        kwargs = {"disabled": trait.read_only, **constructor_kwargs}

        return cls(**kwargs)

    def create_widgets_for_model_cls(
        self, model_cls: Type[traitlets.HasTraits], ctx: ViewFactoryContext
    ) -> Dict[str, widgets.Widget]:
        """Return a mapping from name to widget for a given model class

        :param model_cls: model class object
        :param ctx: factory context
        :return:
        """
        if model_cls in self._visited:
            raise ValueError(f"Already visited {model_cls!r}")

        self._visited.add(model_cls)

        model_widgets = {}
        for name, widget in self.iter_widgets_for_model(model_cls, ctx):
            self.logger.info(f"Created widget {widget} for trait {name!r}")
            model_widgets[name] = widget

        return model_widgets

    def iter_traits(
        self, model_cls: Type[traitlets.HasTraits]
    ) -> Iterator[Tuple[str, traitlets.TraitType]]:
        """Iterate over the name, trait pairs of the model class

        :param model_cls: model class object
        :return:
        """
        yield from model_cls.class_traits().items()

    def iter_widgets_for_model(
        self, model_cls: Type[traitlets.HasTraits], ctx: ViewFactoryContext
    ) -> Iterator[Tuple[str, widgets.Widget]]:
        """Yield (name, widget) pairs corresponding to the traits defined by a model class

        :param model_cls: model class object
        :param ctx: factory context
        :return:
        """
        for name, trait in self.iter_traits(model_cls):
            trait_ctx = ctx.follow(name)

            if not self.can_visit_trait(model_cls, trait, trait_ctx):
                continue

            # Set description only if not set by tag
            # Required because metadata field takes priority over tag metadata
            description = trait.metadata.get("description", trait_ctx.display_name)

            try:
                widget = self.create_trait_view(
                    trait, {"description": description}, ctx
                )

            except:
                ctx.logger.exception(
                    f"Unable to render trait {name!r} ({type(trait).__qualname__})"
                )
                continue

            yield name, widget

    def resolve(self, name_or_cls: Union[str, Type[T]]) -> Type[T]:
        """Resolve the reference to a class

        :param name_or_cls: name of class, or class itself
        :return:
        """
        if isinstance(name_or_cls, str):
            return self._namespace[name_or_cls]
        return name_or_cls
