"""module docstring"""
import sys
import logging

from datetime import datetime
from PyQt5.QtWidgets import QMainWindow, qApp, QMessageBox
from PyQt5.QtGui import QStandardItemModel, QStandardItem, QBrush, QColor
from PyQt5.QtCore import pyqtSlot, Qt
from main_window_conv import Ui_MainClientWindow
from add_contact import AddContactDialog
from del_contact import DelContactDialog
from common.errors import ServerError

sys.path.append('../')
logger = logging.getLogger('client_src')


class ClientMainWindow(QMainWindow):
    """Класс основного окна"""

    def __init__(self, database, transport):
        super().__init__()
        # основные переменные
        self.database = database
        self.transport = transport

        # Загружаем конфигурацию окна из дизайнера
        self.ui = Ui_MainClientWindow()
        self.ui.setupUi(self)

        # Кнопка "Выход"
        self.ui.menu_exit.triggered.connect(qApp.exit)

        # Кнопка отправить сообщение
        self.ui.btn_send.clicked.connect(self.send_message)

        # "добавить контакт"
        self.ui.btn_add_contact.clicked.connect(self.add_contact_window)
        self.ui.menu_add_contact.triggered.connect(self.add_contact_window)

        # Удалить контакт
        self.ui.btn_remove_contact.clicked.connect(self.delete_contact_window)
        self.ui.menu_del_contact.triggered.connect(self.delete_contact_window)

        # Дополнительные требующиеся атрибуты
        self.contacts_model = None
        self.history_model = None
        self.messages = QMessageBox()
        self.current_chat = None
        self.ui.list_messages.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        self.ui.list_messages.setWordWrap(True)

        # Даблклик по листу контактов отправляется в обработчик
        self.ui.list_contacts.doubleClicked.connect(self.select_active_user)

        # переключатель темы
        self.ui.theme_switch.clicked.connect(self.change_theme)
        self.color_set = 0

        self.apply_theme()

        self.clients_list_update()
        self.set_disabled_input()
        self.show()

    def set_theme(self, config):
        self.color_set = int(config['SETTINGS']['theme'][0])
        self.apply_theme()
        if self.color_set:
            self.ui.theme_switch.setChecked(True)

    def apply_theme(self):
        BG_COLOR = ('#c0c0c0', '#10121c')
        LIST_BG_COLOR = ('#c0c0c0', '#1b222b')
        FONT_COLOR = ('#202020', '#c0c0c0')
        MESSAGE_BG_COLOR = ({'green': (180, 255, 180),
                             'red': (255, 180, 180),
                             'grey': (230, 230, 230)
                             },
                            {'green': (50, 90, 133),
                             'red': (33, 34, 46),
                             'grey': (50, 50, 50)
                             })

        # background
        self.ui.centralwidget.setStyleSheet(f'background-color: {BG_COLOR[self.color_set]};')

        # buttons
        for button in [self.ui.btn_add_contact,
                       self.ui.btn_remove_contact,
                       self.ui.btn_send,
                       self.ui.btn_clear]:
            button.setStyleSheet('QPushButton {color: ' + FONT_COLOR[self.color_set] + '}')

        # labels
        for label in [self.ui.label_new_message,
                      self.ui.label_history,
                      self.ui.label_contacts]:
            label.setStyleSheet('QLabel {color: ' + FONT_COLOR[self.color_set] + '}')

        # checkbox
        self.ui.theme_switch.setStyleSheet('QCheckBox {color: ' +
                                           FONT_COLOR[self.color_set] + '}')

        # message color change
        self.message_bg_colors = MESSAGE_BG_COLOR[self.color_set]

        style = '{color: ' + FONT_COLOR[self.color_set] + ';' + \
                ' background-color: ' + LIST_BG_COLOR[self.color_set] + '}'

        # list view
        for q_list in [self.ui.list_messages, self.ui.list_contacts]:
            q_list.setStyleSheet('QListView' + style)

        # text edit field
        self.ui.text_message.setStyleSheet('QTextEdit' + style)

        # menubar
        self.ui.menubar.setStyleSheet('QMenuBar' + style)

        self.history_list_update()

    def change_theme(self):
        self.color_set = self.ui.theme_switch.isChecked()
        self.apply_theme()

    # Деактивировать поля ввода
    def set_disabled_input(self):
        # Надпись  - получатель.
        self.ui.label_new_message.setText('Для выбора получателя дважды кликните на нем в окне контактов.')
        self.ui.text_message.clear()
        if self.history_model:
            self.history_model.clear()

        # Поле ввода и кнопка отправки неактивны до выбора получателя.
        self.ui.btn_clear.setDisabled(True)
        self.ui.btn_send.setDisabled(True)
        self.ui.text_message.setDisabled(True)

    # Заполняем историю сообщений.
    def history_list_update(self):
        # Получаем историю сортированную по дате
        message_list = sorted(self.database.get_history(self.current_chat), key=lambda item: item[3])
        # Если модель не создана, создадим.
        if not self.history_model:
            self.history_model = QStandardItemModel()
            self.ui.list_messages.setModel(self.history_model)

        # Очистим от старых записей
        self.history_model.clear()

        # Заполнение модели записями, так-же стоит разделить входящие и исходящие выравниванием и разным фоном.
        # Записи в обратном порядке, поэтому выбираем их с конца и не более 20
        start_index = max(-20, -len(message_list))

        if message_list:
            date_prev = message_list[start_index][3].date()
            self.history_model.appendRow(self.get_date_qstandarditem(date_prev.strftime('%d-%m-%y')))

            for item in message_list[start_index:]:
                date_curr = item[3].date()
                if date_curr != date_prev:
                    date_prev = date_curr
                    self.history_model.appendRow(self.get_date_qstandarditem(date_prev.strftime('%d-%m-%y')))

                hours_minutes = item[3].strftime('%H:%M')

                mess = QStandardItem()
                mess.setEditable(False)
                text = f'{item[2]}   {hours_minutes}'
                mess.setText(text)
                if item[1] == 'in':
                    mess.setBackground(QBrush(QColor(*self.message_bg_colors['red'])))
                    mess.setTextAlignment(Qt.AlignLeft)
                else:
                    mess.setTextAlignment(Qt.AlignRight)
                    mess.setBackground(QBrush(QColor(*self.message_bg_colors['green'])))
                self.history_model.appendRow(mess)
            self.ui.list_messages.scrollToBottom()

    def get_date_qstandarditem(self, date: datetime) -> QStandardItem:
        date_item = QStandardItem(str(date))
        date_item.setTextAlignment(Qt.AlignCenter)
        date_item.setBackground(QBrush(QColor(*self.message_bg_colors['grey'])))
        return date_item

    # Функция обработчик даблклика по контакту
    def select_active_user(self):
        # Выбранный пользователем (даблклик) находится в выделеном элементе в QListView
        self.current_chat = self.ui.list_contacts.currentIndex().data()
        # вызываем основную функцию
        self.set_active_user()

    # Функция устанавливающяя активного собеседника
    def set_active_user(self):
        # Ставим надпись и активируем кнопки
        self.ui.label_new_message.setText(f'Введите сообщенние для {self.current_chat}:')
        self.ui.btn_clear.setDisabled(False)
        self.ui.btn_send.setDisabled(False)
        self.ui.text_message.setDisabled(False)

        # Заполняем окно историю сообщений по требуемому пользователю.
        self.history_list_update()

    # Функция обновляющяя контакт лист
    def clients_list_update(self):
        contacts_list = self.database.get_contacts()
        self.contacts_model = QStandardItemModel()
        for i in sorted(contacts_list):
            item = QStandardItem(i)
            item.setEditable(False)
            self.contacts_model.appendRow(item)
        self.ui.list_contacts.setModel(self.contacts_model)

    # Функция добавления контакта
    def add_contact_window(self):
        global select_dialog
        select_dialog = AddContactDialog(self.transport, self.database)
        select_dialog.btn_ok.clicked.connect(lambda: self.add_contact_action(select_dialog))
        select_dialog.show()

    # Функция - обработчик добавления, сообщает серверу, обновляет таблицу и список контактов
    def add_contact_action(self, item):
        new_contact = item.selector.currentText()
        self.add_contact(new_contact)
        item.close()

    # Функция добавляющяя контакт в базы
    def add_contact(self, new_contact):
        try:
            self.transport.add_contact(new_contact)
        except ServerError as err:
            self.messages.critical(self, 'Ошибка сервера', err.text)
        except OSError as err:
            if err.errno:
                self.messages.critical(self, 'Ошибка', 'Потеряно соединение с сервером!')
                self.close()
            self.messages.critical(self, 'Ошибка', 'Таймаут соединения!')
        else:
            self.database.add_contact(new_contact)
            new_contact = QStandardItem(new_contact)
            new_contact.setEditable(False)
            self.contacts_model.appendRow(new_contact)
            logger.info(f'Успешно добавлен контакт {new_contact}')
            self.messages.information(self, 'Успех', 'Контакт успешно добавлен.')

    # Функция удаления контакта
    def delete_contact_window(self):
        global remove_dialog
        remove_dialog = DelContactDialog(self.database)
        remove_dialog.btn_ok.clicked.connect(lambda: self.delete_contact(remove_dialog))
        remove_dialog.show()

    # Функция обработчик удаления контакта, сообщает на сервер, обновляет таблицу контактов
    def delete_contact(self, item):
        selected = item.selector.currentText()
        try:
            self.transport.remove_contact(selected)
        except ServerError as err:
            self.messages.critical(self, 'Ошибка сервера', err.text)
        except OSError as err:
            if err.errno:
                self.messages.critical(self, 'Ошибка', 'Потеряно соединение с сервером!')
                self.close()
            self.messages.critical(self, 'Ошибка', 'Таймаут соединения!')
        else:
            self.database.del_contact(selected)
            self.clients_list_update()
            logger.info(f'Успешно удалён контакт {selected}')
            self.messages.information(self, 'Успех', 'Контакт успешно удалён.')
            item.close()
            # Если удалён активный пользователь, то деактивируем поля ввода.
            if selected == self.current_chat:
                self.current_chat = None
                self.set_disabled_input()

    # Функция отправки собщения пользователю.
    def send_message(self):
        # Текст в поле, проверяем что поле не пустое затем забирается сообщение и поле очищается
        message_text = self.ui.text_message.toPlainText().strip()
        self.ui.text_message.clear()
        if not message_text:
            return
        try:
            self.transport.send_message(self.current_chat, message_text)
        except ServerError as err:
            self.messages.critical(self, 'Ошибка', err.text)
        except (ConnectionResetError, ConnectionAbortedError):
            self.messages.critical(self, 'Ошибка', 'Потеряно соединение с сервером!')
            self.close()
        except OSError as err:
            if err.errno:
                self.messages.critical(self, 'Ошибка', 'Потеряно соединение с сервером!')
                self.close()
            self.messages.critical(self, 'Ошибка', 'Таймаут соединения!')
        else:
            self.database.save_message(self.current_chat, 'out', message_text)
            logger.debug(f'Отправлено сообщение для {self.current_chat}: {message_text}')
            self.history_list_update()

    # Слот приёма нового сообщений
    @pyqtSlot(str)
    def message(self, sender):
        if sender == self.current_chat:
            self.history_list_update()
        else:
            # Проверим есть ли такой пользователь у нас в контактах:
            if self.database.check_contact(sender):
                # Если есть, спрашиваем и желании открыть с ним чат и открываем при желании
                answer = self.messages.question(
                    self, 'Новое сообщение',
                    f'Получено новое сообщение от {sender}, открыть чат с ним?',
                    QMessageBox.Yes,
                    QMessageBox.No
                )
                if answer == QMessageBox.Yes:
                    self.current_chat = sender
                    self.set_active_user()
            else:
                print('NO')
                # Раз нету,спрашиваем хотим ли добавить юзера в контакты.
                if self.messages.question(self, 'Новое сообщение',
                                          f'Получено новое сообщение от {sender}.\n '
                                          f'Данного пользователя нет в вашем контакт-листе.\n '
                                          f'Добавить в контакты и открыть чат с ним?',
                                          QMessageBox.Yes,
                                          QMessageBox.No
                                          ) == QMessageBox.Yes:
                    self.add_contact(sender)
                    self.current_chat = sender
                    self.set_active_user()

    # Слот потери соединения
    # Выдаёт сообщение о ошибке и завершает работу приложения
    @pyqtSlot()
    def connection_lost(self):
        self.messages.warning(self, 'Сбой соединения', 'Потеряно соединение с сервером. ')
        self.close()

    def make_connection(self, trans_obj):
        trans_obj.new_message.connect(self.message)
        trans_obj.connection_lost.connect(self.connection_lost)
