"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const METRICS_NAMESPACE = 'ConstructHub/CatalogBuilder';
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    // determine if this is a request to rebuild the catalog (basically, an empty event)
    const rebuild = !(event === null || event === void 0 ? void 0 : event.package);
    if (rebuild) {
        console.log('Requesting catalog rebuild (empty event)');
    }
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    let data;
    if (!rebuild) {
        console.log('Loading existing catalog...');
        data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
            .catch((err) => err.code !== 'NoSuchKey'
            ? Promise.reject(err)
            : Promise.resolve({ /* no data */}));
    }
    // if event is empty, we're doing a full rebuild
    if (!(data === null || data === void 0 ? void 0 : data.Body) || rebuild) {
        console.log('Catalog not found. Recreating...');
        const failures = {};
        for await (const { Key: pkgKey } of relevantObjects(BUCKET_NAME)) {
            try {
                await appendPackage(packages, pkgKey, BUCKET_NAME, denyList);
            }
            catch (e) {
                failures[pkgKey] = e;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setNamespace(METRICS_NAMESPACE);
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric('FailedPackagesOnRecreation', failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    else {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
        console.log('Registering new packages...');
        // note that we intentionally don't catch errors here to let these
        // event go to the DLQ for manual inspection.
        await appendPackage(packages, event.package.key, BUCKET_NAME, denyList);
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setNamespace(METRICS_NAMESPACE);
        metrics.putMetric('RegisteredPackagesMajorVersion', catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Clean up existing entries if necessary. In particular, remove the license texts as they make
    // the catalog unnecessarily large, and may hinder some search queries' result quality.
    for (const entry of catalog.packages) {
        if (entry.metadata) {
            delete entry.metadata.licenseText;
        }
    }
    // Upload the result to S3 and exit.
    return aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: 'public, max-age=300',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: constants.STORAGE_KEY_PREFIX };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            // We only register packages if they have AT LEAST docs in one language.
            const tsDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_TYPESCRIPT}`;
            const pyDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_PYTHON}`;
            const javaDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_JAVA}`;
            if (!(await aws.s3ObjectExists(bucket, tsDocs)) &&
                !(await aws.s3ObjectExists(bucket, pyDocs)) &&
                !(await aws.s3ObjectExists(bucket, javaDocs))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, pkgKey, bucketName, denyList) {
    var _a, _b, _c;
    console.log(`Processing key: ${pkgKey}`);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    if (found != null && version.compare(found.version) <= 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Checking if ${packageName}@${version.version} matches a deny list rule`);
    const blocked = denyList.lookup(packageName, version.version);
    if (blocked) {
        console.log(`Skipping ${packageName}@${version.version} because it is blocked by the deny list rule: ${JSON.stringify(blocked)}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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