"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const stream_1 = require("stream");
const zip = require("zlib");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const tar = require("tar-stream");
const catalog_builder_lambda_1 = require("../../../backend/catalog-builder/catalog-builder.lambda");
const aws = require("../../../backend/shared/aws.lambda-shared");
const constants = require("../../../backend/shared/constants.lambda-shared");
let mockBucketName;
beforeEach((done) => {
    process.env.BUCKET_NAME = mockBucketName = crypto_1.randomBytes(16).toString('base64');
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    aws.reset();
    process.env.BUCKET_NAME = mockBucketName = undefined;
    done();
});
test('no indexed packages', () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
        }
        catch (e) {
            return cb(e);
        }
        return cb(new NoSuchKeyError());
    });
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
            expect(req.ContinuationToken).toBeUndefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, {});
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('initial build', () => {
    // GIVEN
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            return cb(null, { Body: JSON.stringify(npmMetadata) });
        }
        const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    const mockFirstPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
    ];
    const mockSecondPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.0.0${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.ASSEMBLY_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre${constants.PACKAGE_KEY_SUFFIX}` },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '3');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '2.0.0-pre',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
test('incremental build', () => {
    const npmMetadata = { date: 'Thu, 17 Jun 2021 01:52:04 GMT' };
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
        }
        catch (e) {
            return cb(e);
        }
        if (req.Key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            return cb(null, { Body: JSON.stringify(npmMetadata) });
        }
        const matches = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`).exec(req.Key);
        if (matches != null) {
            mockNpmPackage(matches[1], matches[2]).then((pack) => cb(null, { Body: pack }), cb);
        }
        else if (req.Key === constants.CATALOG_KEY) {
            return cb(null, {
                Body: JSON.stringify({
                    packages: [
                        {
                            description: 'Package @scope/package, version 2.3.4',
                            languages: { foo: 'bar' },
                            major: 2,
                            metadata: npmMetadata,
                            name: '@scope/package',
                            version: '2.3.4',
                        },
                        {
                            description: 'Package name, version 1.0.0',
                            languages: { foo: 'bar' },
                            major: 1,
                            metadata: npmMetadata,
                            name: 'name',
                            version: '1.0.0',
                        },
                        {
                            description: 'Package name, version 2.0.0-pre.10',
                            languages: { foo: 'bar' },
                            major: 2,
                            metadata: npmMetadata,
                            name: 'name',
                            version: '2.0.0-pre.10',
                        },
                    ],
                    updatedAt: new Date().toISOString(),
                }, null, 2),
            });
        }
        else {
            return cb(new NoSuchKeyError());
        }
    });
    const mockFirstPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v1.2.3${constants.PACKAGE_KEY_SUFFIX}` },
    ];
    const mockSecondPage = [
        { Key: `${constants.STORAGE_KEY_PREFIX}@scope/package/v2.0.5${constants.PACKAGE_KEY_SUFFIX}` },
        { Key: `${constants.STORAGE_KEY_PREFIX}name/v2.0.0-pre.1${constants.PACKAGE_KEY_SUFFIX}` },
    ];
    AWSMock.mock('S3', 'listObjectsV2', (req, cb) => {
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Prefix).toBe(constants.STORAGE_KEY_PREFIX);
        }
        catch (e) {
            return cb(e);
        }
        if (req.ContinuationToken == null) {
            return cb(null, { Contents: mockFirstPage, NextContinuationToken: 'next' });
        }
        try {
            expect(req.ContinuationToken).toBe('next');
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Contents: mockSecondPage });
    });
    const mockPutObjectResult = {};
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect(req.Key).toBe(constants.CATALOG_KEY);
            expect(req.ContentType).toBe('text/json');
            expect(req.Metadata).toHaveProperty('Package-Count', '4');
            const body = JSON.parse((_b = (_a = req.Body) === null || _a === void 0 ? void 0 : _a.toString('utf-8')) !== null && _b !== void 0 ? _b : 'null');
            expect(body.packages).toEqual([
                {
                    description: 'Package @scope/package, version 2.3.4',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '2.3.4',
                },
                {
                    description: 'Package @scope/package, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: '@scope/package',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 1.2.3',
                    languages: { foo: 'bar' },
                    major: 1,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '1.2.3',
                },
                {
                    description: 'Package name, version 2.0.0-pre.10',
                    languages: { foo: 'bar' },
                    major: 2,
                    metadata: npmMetadata,
                    name: 'name',
                    version: '2.0.0-pre.10',
                },
            ]);
            expect(Date.parse(body.updatedAt)).toBeDefined();
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, mockPutObjectResult);
    });
    // WHEN
    const result = catalog_builder_lambda_1.handler({}, { /* context */});
    // THEN
    return expect(result).resolves.toBe(mockPutObjectResult);
});
class NoSuchKeyError extends Error {
    constructor() {
        super(...arguments);
        this.code = 'NoSuchKey';
        this.time = new Date();
    }
}
function mockNpmPackage(name, version) {
    const packageJson = {
        name,
        version,
        description: `Package ${name}, version ${version}`,
        jsii: {
            targets: { foo: 'bar' },
        },
    };
    const tarball = tar.pack();
    tarball.entry({ name: 'package/ignore-me.txt' }, 'Ignore Me!');
    tarball.entry({ name: 'package/package.json' }, JSON.stringify(packageJson, null, 2));
    tarball.finalize();
    const gzip = zip.createGzip();
    tarball.pipe(gzip);
    const passthrough = new stream_1.PassThrough();
    gzip.pipe(passthrough);
    return new Promise((ok) => {
        const chunks = new Array();
        passthrough.on('data', (chunk) => chunks.push(Buffer.from(chunk)));
        passthrough.once('end', () => {
            ok(Buffer.concat(chunks));
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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