"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const KEY_FORMAT_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                   ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    var _a, _b, _c;
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    if (!event.rebuild) {
        console.log('Loading existing catalog...');
        const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
            .catch((err) => err.code !== 'NoSuchKey'
            ? Promise.reject(err)
            : Promise.resolve({ /* no data */}));
        if (data.Body != null) {
            const catalog = JSON.parse(data.Body.toString('utf-8'));
            for (const info of catalog.packages) {
                if (!packages.has(info.name)) {
                    packages.set(info.name, new Map());
                }
                packages.get(info.name).set(info.major, info);
            }
        }
    }
    else {
        console.log('Rebuild requested, ignoring existing catalog...');
    }
    console.log('Listing existing objects...');
    for await (const object of relevantObjects(BUCKET_NAME)) {
        // Ensure we don't already have a more recent version tracked for this package-major.
        const [, packageName, versionStr] = object.Key.match(KEY_FORMAT_REGEX);
        const version = new semver_1.SemVer(versionStr);
        const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
        if (found != null && version.compare(found.version) <= 0) {
            console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
            continue;
        }
        console.log(`Registering ${packageName}@${version}`);
        // Donwload the tarball to inspect the `package.json` data therein.
        const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: object.Key }).promise();
        const npmMetadataKey = object.Key.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
        const npmMetadataResponse = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: npmMetadataKey }).promise();
        const manifest = await new Promise((ok, ko) => {
            zlib_1.gunzip(Buffer.from(data.Body), (err, tar) => {
                if (err) {
                    return ko(err);
                }
                tar_stream_1.extract()
                    .on('entry', (header, stream, next) => {
                    if (header.name !== 'package/package.json') {
                        // Not the file we are looking for, skip ahead (next run-loop tick).
                        return setImmediate(next);
                    }
                    const chunks = new Array();
                    return stream
                        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('end', () => {
                        ok(Buffer.concat(chunks));
                        next();
                    })
                        .resume();
                })
                    .once('finish', () => {
                    ko(new Error('Could not find package/package.json in tarball!'));
                })
                    .write(tar, (writeErr) => {
                    if (writeErr) {
                        ko(writeErr);
                    }
                });
            });
        });
        // Add the PackageInfo into the working set
        const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
        const npmMetadata = JSON.parse((_c = (_b = npmMetadataResponse === null || npmMetadataResponse === void 0 ? void 0 : npmMetadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
        const major = new semver_1.SemVer(pkgMetadata.version).major;
        if (!packages.has(pkgMetadata.name)) {
            packages.set(pkgMetadata.name, new Map());
        }
        packages.get(pkgMetadata.name).set(major, {
            author: pkgMetadata.author,
            description: pkgMetadata.description,
            keywords: pkgMetadata.keywords,
            languages: pkgMetadata.jsii.targets,
            license: pkgMetadata.license,
            major,
            metadata: npmMetadata,
            name: pkgMetadata.name,
            time: pkgMetadata.time,
            version: pkgMetadata.version,
        });
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`Registered ${catalog.packages.length} package major versions`);
    // Upload the result to S3 and exit.
    return aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'text/json',
        Metadata: {
            'Build-Process': event.rebuild ? 'FROM_SCRATCH' : 'INCREMENTAL',
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: constants.STORAGE_KEY_PREFIX };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
//# sourceMappingURL=data:application/json;base64,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