"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CatalogBuilder = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants = require("../shared/constants.lambda-shared");
const catalog_builder_1 = require("./catalog-builder");
/**
 * Builds or re-builds the `catalog.json` object in the designated bucket.
 */
class CatalogBuilder extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const handler = new catalog_builder_1.CatalogBuilder(this, 'Default', {
            deadLetterQueueEnabled: true,
            description: `Creates the catalog.json object in ${props.bucket.bucketName}`,
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
            },
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantReadWrite(handler);
        handler.addEventSource(new aws_lambda_event_sources_1.S3EventSource(props.bucket, {
            events: [aws_s3_1.EventType.OBJECT_CREATED],
            filters: [{ prefix: constants.STORAGE_KEY_PREFIX, suffix: constants.ASSEMBLY_KEY_SUFFIX }],
        }));
        props.monitoring.watchful.watchLambdaFunction('Catalog Builder Function', handler);
        this.alarmDeadLetterQueueNotEmpty = handler.deadLetterQueue.metricApproximateNumberOfMessagesVisible()
            .createAlarm(this, 'DLQAlarm', {
            alarmDescription: 'The catalog builder function failed to run',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
}
exports.CatalogBuilder = CatalogBuilder;
//# sourceMappingURL=data:application/json;base64,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