"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const MARKER_FILE_NAME = 'couchdb-last-transaction-id';
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const initialMarker = await loadLastTransactionMarker(1800000 /* @aws-cdk/cdk initial release was at 1_846_709 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: initialMarker.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = initialMarker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            metrics.setNamespace('ConstructHub/Discovery');
            const startTime = Date.now();
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric('BatchSize', batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric('RelevantPackageVersions', versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // Process all remaining updates
                await Promise.all(versionInfos.map(async (infos) => {
                    const before = Date.now();
                    await processUpdatedVersion(infos);
                    metrics.putMetric('StagingTime', Date.now() - before, aws_embedded_metrics_1.Unit.Milliseconds);
                }));
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                updatedMarker = lastSeq;
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric('RemainingTime', context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok({ initialMarker, updatedMarker });
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric('BatchProcessingTime', Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok({ initialMarker, updatedMarker });
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: MARKER_FILE_NAME,
            }).promise();
            const marker = Number.parseInt(response.Body.toString('utf-8'), 10);
            console.log(`Read last transaction marker: ${marker}`);
            return marker;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${MARKER_FILE_NAME}) does not exist, starting from the default (${defaultValue})`);
            return defaultValue;
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return putObject(MARKER_FILE_NAME, sequence.toFixed(), { ContentType: 'text/plain' });
    }
    //#endregion
    //#region Business Logic
    async function processUpdatedVersion({ infos, modified, seq }) {
        try {
            // Download the tarball
            const tarball = await httpGet(infos.dist.tarball);
            // Store the tarball into the staging bucket
            // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
            // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
            const stagingKey = `${constants.STAGED_KEY_PREFIX}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
            await putObject(stagingKey, tarball, {
                ContentType: 'application/x-gtar',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                    'Origin-Integrity': infos.dist.shasum,
                    'Origin-URI': infos.dist.tarball,
                    'Sequence': seq.toFixed(),
                },
            });
            // Prepare SQS message for ingestion
            const messageBase = {
                tarballUri: `s3://${stagingBucket}/${stagingKey}`,
                metadata: {
                    dist: infos.dist.tarball,
                    seq: seq.toFixed(),
                },
                time: modified.toUTCString(),
            };
            const message = {
                ...messageBase,
                integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
            };
            // Send the SQS message out
            await aws.sqs().sendMessage({
                MessageBody: JSON.stringify(message, null, 2),
                QueueUrl: queueUrl,
            }).promise();
        }
        catch (err) {
            // Something failed, store the payload in the problem prefix, and move on.
            console.error(`[${seq}] Failed processing, logging error to S3 and resuming work. ${infos.name}@${infos.version}: ${err}`);
            await putObject(`${constants.FAILED_KEY_PREFIX}${seq}`, JSON.stringify({ ...infos, _construct_hub_failure_reason: err }, null, 2), {
                ContentType: 'text/json',
                Metadata: {
                    'Modified-At': modified.toISOString(),
                },
            });
        }
    }
    //#endregion
    //#region Asynchronous Primitives
    /**
     * Makes an HTTP GET request, and returns the resulting payload.
     *
     * @param url the URL to get.
     *
     * @returns a Buffer containing the received data.
     */
    function httpGet(url) {
        return new Promise((ok, ko) => {
            https.get(url, (response) => {
                if (response.statusCode !== 200) {
                    throw new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`);
                }
                let body = Buffer.alloc(0);
                response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
                response.once('close', () => ok(body));
                response.once('error', ko);
            });
        });
    }
    /**
     * Puts an object in the staging bucket, with standardized object metadata.
     *
     * @param key  the key for the object to be put.
     * @param body the body of the object to be put.
     * @param opts any other options to use when sending the S3 request.
     *
     * @returns the result of the S3 request.
     */
    function putObject(key, body, opts = {}) {
        return aws.s3().putObject({
            Bucket: stagingBucket,
            Key: key,
            Body: body,
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
                ...opts.Metadata,
            },
            ...opts,
        }).promise();
    }
    //#endregion
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics) {
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric('UnprocessableEntity', 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric('UnprocessableEntity', 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric('UnprocessableEntity', 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric('UnprocessableEntity', 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        let latestModified;
        for (const [version, modified] of sortedUpdates) {
            if (latestModified == null || latestModified.getTime() === modified.getTime()) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isRelevantPackageVersion(infos)) {
                    metrics.putMetric('PackageVersionAge', Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    result.push({ infos, modified, seq: change.seq });
                }
                else {
                    console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
                }
                latestModified = modified;
            }
        }
    }
    return result;
    function isRelevantPackageVersion(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
}
//# sourceMappingURL=data:application/json;base64,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