"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants_lambda_shared_1 = require("../shared/constants.lambda-shared");
const discovery_1 = require("./discovery");
/**
 * This discovery function periodically scans the CouchDB replica of npmjs.com
 * to discover newly published packages that are relevant for indexing in the
 * Construct Hub, then notifies the ingestion function about those.
 */
class Discovery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.bucket = new aws_s3_1.Bucket(this, 'StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            lifecycleRules: [
                {
                    prefix: constants_lambda_shared_1.STAGED_KEY_PREFIX,
                    expiration: core_1.Duration.days(30),
                },
            ],
        });
        // Note: the handler is designed to stop processing more batches about 2 minutes ahead of the timeout.
        const timeout = core_1.Duration.minutes(15);
        const lambda = new discovery_1.Discovery(this, 'Default', {
            description: 'Periodically query npm.js index for new construct libraries',
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout,
            environment: {
                BUCKET_NAME: this.bucket.bucketName,
                QUEUE_URL: props.queue.queueUrl,
            },
        });
        this.bucket.grantReadWrite(lambda);
        props.queue.grantSendMessages(lambda);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(timeout),
            targets: [new aws_events_targets_1.LambdaFunction(lambda)],
        });
        props.monitoring.watchful.watchLambdaFunction('Discovery Function', lambda);
        this.alarmErrors = lambda.metricErrors({ period: core_1.Duration.minutes(15) }).createAlarm(this, 'ErrorsAlarm', {
            alarmDescription: 'The discovery function (on npmjs.com) failed to run',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
        this.alarmNoInvocations = lambda.metricInvocations({ period: core_1.Duration.minutes(15) })
            .createAlarm(this, 'NoInvocationsAlarm', {
            alarmDescription: 'The discovery function (on npmjs.com) is not running as scheduled',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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