"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const url_1 = require("url");
const zlib_1 = require("zlib");
const spec_1 = require("@jsii/spec");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
async function handler(event, context) {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws.s3().getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        }).promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        const tar = await gunzip(Buffer.from(tarball.Body));
        const dotJsii = await new Promise((ok, ko) => {
            tar_stream_1.extract()
                .on('entry', (headers, stream, next) => {
                if (headers.name !== 'package/.jsii') {
                    // Skip on next runLoop iteration so we avoid filling the stack.
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('error', ko)
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('error', ko)
                .once('close', () => ko(new Error('No .jsii file found in tarball!')))
                .write(tar, (err) => {
                if (err != null) {
                    ko(err);
                }
            });
        });
        const metadata = { date: payload.time };
        const { name: packageName, version: packageVersion } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
        const assemblyKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`;
        console.log(`Writing assembly at ${assemblyKey}`);
        const packageKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.PACKAGE_KEY_SUFFIX}`;
        console.log(`Writing package at  ${packageKey}`);
        const metadataKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.METADATA_KEY_SUFFIX}`;
        console.log(`Writing metadata at  ${metadataKey}`);
        const [assembly, pkg, storedMetadata] = await Promise.all([
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: assemblyKey,
                Body: dotJsii,
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                ContentType: 'application/x-gtar',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
        ]);
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        result.push(created);
    }
    return result;
}
exports.handler = handler;
function gunzip(data) {
    const chunks = new Array();
    return new Promise((ok, ko) => zlib_1.createGunzip()
        .once('error', ko)
        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
        .once('end', () => ok(Buffer.concat(chunks)))
        .end(data));
}
//# sourceMappingURL=data:application/json;base64,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