"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants = require("../shared/constants.lambda-shared");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const lambda = new transliterator_1.Transliterator(this, 'Default', {
            deadLetterQueueEnabled: true,
            description: 'Creates transliterated assemblies from jsii-enabled npm packages',
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(15),
        });
        // The handler reads & writes to this bucket.
        props.bucket.grantReadWrite(lambda);
        // Creating the event chaining
        lambda.addEventSource(new aws_lambda_event_sources_1.S3EventSource(props.bucket, {
            events: [aws_s3_1.EventType.OBJECT_CREATED],
            filters: [{ prefix: constants.STORAGE_KEY_PREFIX, suffix: constants.PACKAGE_KEY_SUFFIX }],
        }));
        props.monitoring.watchful.watchLambdaFunction('Transliterator Function', lambda);
        this.alarmDeadLetterQueueNotEmpty = lambda.deadLetterQueue.metricApproximateNumberOfMessagesVisible()
            .createAlarm(this, 'DLQAlarm', {
            alarmDescription: 'The transliteration function failed for one or more packages',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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