"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const url_1 = require("url");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * (experimental) Elasticsearch version.
 *
 * @experimental
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * (experimental) Custom Elasticsearch version.
     *
     * @param version custom version number.
     * @experimental
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
/**
 * (experimental) AWS Elasticsearch 1.5.
 *
 * @experimental
 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/**
 * (experimental) AWS Elasticsearch 2.3.
 *
 * @experimental
 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/**
 * (experimental) AWS Elasticsearch 5.1.
 *
 * @experimental
 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/**
 * (experimental) AWS Elasticsearch 5.3.
 *
 * @experimental
 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/**
 * (experimental) AWS Elasticsearch 5.5.
 *
 * @experimental
 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/**
 * (experimental) AWS Elasticsearch 5.6.
 *
 * @experimental
 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/**
 * (experimental) AWS Elasticsearch 6.0.
 *
 * @experimental
 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/**
 * (experimental) AWS Elasticsearch 6.2.
 *
 * @experimental
 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/**
 * (experimental) AWS Elasticsearch 6.3.
 *
 * @experimental
 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/**
 * (experimental) AWS Elasticsearch 6.4.
 *
 * @experimental
 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/**
 * (experimental) AWS Elasticsearch 6.5.
 *
 * @experimental
 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/**
 * (experimental) AWS Elasticsearch 6.7.
 *
 * @experimental
 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/**
 * (experimental) AWS Elasticsearch 6.8.
 *
 * @experimental
 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/**
 * (experimental) AWS Elasticsearch 7.1.
 *
 * @experimental
 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/**
 * (experimental) AWS Elasticsearch 7.4.
 *
 * @experimental
 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/**
 * (experimental) AWS Elasticsearch 7.7.
 *
 * @experimental
 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * (experimental) The minimum TLS version required for traffic to the domain.
 *
 * @experimental
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * (experimental) Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Return the given named metric for this Domain.
     *
     * @experimental
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensions: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        });
    }
    /**
     * (experimental) Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @experimental
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @experimental
     */
    metricClusterIndexWriteBlocked(props) {
        return this.metric('ClusterIndexWriteBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @experimental
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * (experimental) Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * (experimental) Provides an Elasticsearch domain.
 *
 * @experimental
 */
class Domain extends DomainBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29;
        super(scope, id, {
            physicalName: props.domainName,
        });
        const defaultInstanceType = 'r5.large.elasticsearch';
        const dedicatedMasterType = (_c = (_b = (_a = props.capacity) === null || _a === void 0 ? void 0 : _a.masterNodeInstanceType) === null || _b === void 0 ? void 0 : _b.toLowerCase()) !== null && _c !== void 0 ? _c : defaultInstanceType;
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = dedicatedMasterCount > 0;
        const instanceType = (_h = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType) === null || _g === void 0 ? void 0 : _g.toLowerCase()) !== null && _h !== void 0 ? _h : defaultInstanceType;
        const instanceCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.dataNodes) !== null && _k !== void 0 ? _k : 1;
        const availabilityZoneCount = (_m = (_l = props.zoneAwareness) === null || _l === void 0 ? void 0 : _l.availabilityZoneCount) !== null && _m !== void 0 ? _m : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_p = (_o = props.zoneAwareness) === null || _o === void 0 ? void 0 : _o.enabled) !== null && _p !== void 0 ? _p : ((_q = props.zoneAwareness) === null || _q === void 0 ? void 0 : _q.availabilityZoneCount) != null;
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (props.vpcOptions != null && zoneAwarenessEnabled &&
            new Set((_r = props.vpcOptions) === null || _r === void 0 ? void 0 : _r.subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        ;
        if ([dedicatedMasterType, instanceType].some(t => !t.endsWith('.elasticsearch'))) {
            throw new Error('Master and data node instance types must end with ".elasticsearch".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_s = props.useUnsignedBasicAuth) !== null && _s !== void 0 ? _s : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_t = props.encryptionAtRest) === null || _t === void 0 ? void 0 : _t.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.Anyone()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_u = props.fineGrainedAccessControl) === null || _u === void 0 ? void 0 : _u.masterUserArn;
        const masterUserNameProps = (_v = props.fineGrainedAccessControl) === null || _v === void 0 ? void 0 : _v.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_y = (_x = props.encryptionAtRest) === null || _x === void 0 ? void 0 : _x.enabled) !== null && _y !== void 0 ? _y : (((_z = props.encryptionAtRest) === null || _z === void 0 ? void 0 : _z.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_0 = props.nodeToNodeEncryption) !== null && _0 !== void 0 ? _0 : unsignedBasicAuthEnabled;
        const volumeSize = (_2 = (_1 = props.ebs) === null || _1 === void 0 ? void 0 : _1.volumeSize) !== null && _2 !== void 0 ? _2 : 10;
        const volumeType = (_4 = (_3 = props.ebs) === null || _3 === void 0 ? void 0 : _3.volumeType) !== null && _4 !== void 0 ? _4 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_6 = (_5 = props.ebs) === null || _5 === void 0 ? void 0 : _5.enabled) !== null && _6 !== void 0 ? _6 : true;
        const enforceHttps = (_7 = props.enforceHttps) !== null && _7 !== void 0 ? _7 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if (((_8 = props.logging) === null || _8 === void 0 ? void 0 : _8.slowIndexLogEnabled) || ((_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.appLogEnabled)
                || ((_10 = props.logging) === null || _10 === void 0 ? void 0 : _10.slowSearchLogEnabled)) {
                throw new Error('Error and slow logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_11 = props.encryptionAtRest) === null || _11 === void 0 ? void 0 : _11.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isInstanceType('i3') && ebsEnabled) {
            throw new Error('I3 instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        // Only R3 and I3 support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3')) {
            throw new Error('EBS volumes are required when using instance types other than r3 or i3.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        let cfnVpcOptions;
        if (props.vpcOptions) {
            cfnVpcOptions = {
                securityGroupIds: props.vpcOptions.securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: props.vpcOptions.subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_13 = props.logging.slowSearchLogGroup) !== null && _13 !== void 0 ? _13 : new logs.LogGroup(scope, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_14 = props.logging) === null || _14 === void 0 ? void 0 : _14.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_15 = props.logging.slowIndexLogGroup) !== null && _15 !== void 0 ? _15 : new logs.LogGroup(scope, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_16 = props.logging) === null || _16 === void 0 ? void 0 : _16.appLogEnabled) {
            this.appLogGroup = (_17 = props.logging.appLogGroup) !== null && _17 !== void 0 ? _17 : new logs.LogGroup(scope, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, 'ESLogGroupPolicy', {
                policyName: 'ESLogPolicy',
                policyStatements: [logPolicyStatement],
            });
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_18 = props.ebs) === null || _18 === void 0 ? void 0 : _18.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_20 = (_19 = props.encryptionAtRest) === null || _19 === void 0 ? void 0 : _19.kmsKey) === null || _20 === void 0 ? void 0 : _20.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: {
                ES_APPLICATION_LOGS: {
                    enabled: this.appLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_21 = this.appLogGroup) === null || _21 === void 0 ? void 0 : _21.logGroupArn,
                },
                SEARCH_SLOW_LOGS: {
                    enabled: this.slowSearchLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_22 = this.slowSearchLogGroup) === null || _22 === void 0 ? void 0 : _22.logGroupArn,
                },
                INDEX_SLOW_LOGS: {
                    enabled: this.slowIndexLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_23 = this.slowIndexLogGroup) === null || _23 === void 0 ? void 0 : _23.logGroupArn,
                },
            },
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_24 = props.cognitoKibanaAuth) === null || _24 === void 0 ? void 0 : _24.identityPoolId,
                roleArn: (_25 = props.cognitoKibanaAuth) === null || _25 === void 0 ? void 0 : _25.role.roleArn,
                userPoolId: (_26 = props.cognitoKibanaAuth) === null || _26 === void 0 ? void 0 : _26.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_27 = props.tlsSecurityPolicy) !== null && _27 !== void 0 ? _27 : TLSSecurityPolicy.TLS_1_0,
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_28 = this.masterUserPassword) === null || _28 === void 0 ? void 0 : _28.toString(),
                    },
                }
                : undefined,
        });
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_29 = props.accessPolicies) !== null && _29 !== void 0 ? _29 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @experimental
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @experimental
     */
    static fromDomainAttributes(scope, id, attrs) {
        const { domainArn, domainEndpoint } = attrs;
        const domainName = extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
}
exports.Domain = Domain;
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
//# sourceMappingURL=data:application/json;base64,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