# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dynamodb_session_web']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.21.21,<2.0.0']

setup_kwargs = {
    'name': 'dynamodb-session-web',
    'version': '0.2.3',
    'description': 'Contains the core API for a DynamoDB-backed session',
    'long_description': '# dynamodb-session-web\n\nAn implementation of a "web" session using DynamoDB as backend storage. This project has the following goals:\n* Focus on core session handling concerns, rather than specific Python frameworks.\n* Follow [OWASP Session Management](https://cheatsheetseries.owasp.org/cheatsheets/Session_Management_Cheat_Sheet.html) \nbest practices by default. Specifically, best practices for:\n  * Session ID\n    - [X] Length, value, and entropy: ID is a 32-byte secure random number. \n    - [X] Strict session management: ID generation can only occur within the framework.\n    - [X] Treat as sensitive: A "loggable session ID" is provided as a property in order to make logging the ID easier (if desired).\n  * Timeouts\n    - [X] Absolute session timeout - default of 12 hours\n    - [X] Idle session timeout - default of 2 hours\n    - [X] Manual session timeout - i.e. there\'s delete/clear support\n\n## Usage\n\nRequires a DynamoDB table named `app_session` (can be changed in settings)\n\n### Create session table\n\n```shell\naws dynamodb create-table \\\n    --attribute-definitions \\\n        AttributeName=id,AttributeType=S \\\n    --key-schema "AttributeName=id,KeyType=HASH" \\\n    --provisioned-throughput "ReadCapacityUnits=5,WriteCapacityUnits=5" \\\n    --table-name app_session \n```\n\n### Default Example\n```python\nfrom dynamodb_session_web import SessionManager\n\nsession = SessionManager()\n\n# Create a new session object, get the ID for later use\ninitial_data = session.create()\nsession_id = initial_data.session_id\n\ninitial_data[\'foo\'] = \'bar\'\ninitial_data[\'one\'] = 1\nsession.save(initial_data)\n\nprint(session_id)\n#> \'WaHnSSou4d5Rq0k11vFGafe4sjMrkwiVhNziIWLLwMc\'\nprint(initial_data.loggable_session_id)\n#> \'517286da2682be08dc9975612dc86d65487f0990906656f631d419e64dcda6f41f5e0529c290663be315524a0b35777645e0e827d2e982a048b5e2b4bba4e02b\'\n\nloaded_data = session.load(session_id)\nprint(loaded_data[\'foo\'])\n#> \'bar\'\nprint(loaded_data[\'one\'])\n#> 1\n\nsession.clear(session_id)\n```\n\n### Configurable Timeout and NullSession Response\n```python\nfrom time import sleep\nfrom dynamodb_session_web import SessionManager\n\nsession = SessionManager()\n\n# Create a new session object, get the ID for later use\ninitial_data = session.create()\ninitial_data.idle_timeout_seconds = 30\ninitial_data.absolute_timeout_seconds = 30\nsession_id = initial_data.session_id\n\ninitial_data[\'foo\'] = \'bar\'\nsession.save(initial_data)\n\nsleep(35)\n\nloaded_data = session.load(session_id)\nprint(loaded_data)\n#> <dynamodb_session_web.NullSessionInstance object at 0x109a7da30>\n```\n\n\n### Custom Data Class Example\n```python\nimport json\nfrom dataclasses import asdict, dataclass\n\nfrom dynamodb_session_web import SessionInstanceBase, SessionManager\n\n@dataclass\nclass MySession(SessionInstanceBase):\n    fruit: str = \'\'\n    color: str = \'\'\n\n    def __init__(self, **kwargs):\n        super().__init__(**kwargs)\n\n    def deserialize(self, data):\n        data_dict = json.loads(data)\n        self.fruit = data_dict[\'fruit\']\n        self.color = data_dict[\'color\']\n\n    def serialize(self):\n        return json.dumps(asdict(self))\n\nsession = SessionManager(MySession)\n\n# Create a new session object, get the ID for later use\ninitial_data = session.create()\nsession_id = initial_data.session_id\n\ninitial_data.fruit = \'apple\'\ninitial_data.color = \'red\'\nsession.save(initial_data)\n\nloaded_data = session.load(session_id)\nprint(loaded_data.fruit)\n#> \'apple\'\nprint(loaded_data.color)\n#> \'red\'\n\nsession.clear(session_id)\n```\n\n## Configuration\n\nSeveral behaviors can be configured at the Session Manager level:\n* Custom data class; must provide serialization and deserialization methods (see examples). Defaults to a dictionary.\n* Session ID length. Defaults to 32 bytes.\n* Table name. Defaults to `app_session`\n* DynamoDB URL. Defaults to None (i.e. Boto3 logic).\n* Idle session timeout (in seconds). Defaults to 7200 seconds (2 hours).\n* Absolute session timeout (in seconds). Defaults to 43200 seconds (12 hours).\n\n```python\nfrom dynamodb_session_web import SessionInstanceBase, SessionManager\n\nclass MyCustomDataClass(SessionInstanceBase):\n    def deserialize(self, data: str):\n        pass\n\n    def serialize(self) -> str:\n        pass\n\nSessionManager(\n    MyCustomDataClass,\n    sid_byte_length=128,\n    table_name=\'my-dynamodb-table\',\n    endpoint_url=\'http://localhost:8000\',\n    idle_timeout_seconds=300,\n    absolute_timeout_seconds=3600,\n)\n```\n\nAdditionally, individual session instances can have their own idle and absolute timeouts, specified in seconds:\n\n```python\nfrom dynamodb_session_web import SessionManager\n\nsession = SessionManager()\n\ninstance = session.create()\ninstance.idle_timeout_seconds = 30\ninstance.absolute_timeout_seconds = 30\n```\n\n## Development\n\n### Prerequisites:\n\n* Docker\n\n### Tests\n\nThe integration tests will use the `docker-compose.yml` file to create a local DynamoDB instance.\n\n## Useful Things\n\n### Get a record from local DynamoDB instance\n\n```shell\nexport sid=Jh4f1zvVp9n-YaDbkpZ0Vtru6iCXnERZv40q_ocZ7BA\naws dynamodb query --table-name app_session --endpoint-url http://localhost:8000 \\\n  --key-condition-expression \'id = :id\' \\\n  --expression-attribute-values \'{ ":id": {"S": "\'$sid\'" }}\'\n```\n',
    'author': 'Jason Capriotti',
    'author_email': 'jason.capriotti@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/JCapriotti/dynamodb-session-web',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
