"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PDKNag = exports.PDKNagApp = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cdk_nag_1 = require("cdk-nag");
const CDK_NAG_MESSAGE_TYPES = {
    ERROR: "aws:cdk:error",
    WARNING: "aws:cdk:warning",
};
const CDK_NAG_MESSAGE_TYPES_SET = new Set(Object.values(CDK_NAG_MESSAGE_TYPES));
/**
 * @inheritDoc
 */
class PDKNagApp extends aws_cdk_lib_1.App {
    constructor(props) {
        super(props);
        this._nagResults = [];
        this.failOnError = props?.failOnError ?? false;
        this.failOnWarning = props?.failOnWarning ?? false;
    }
    synth(options) {
        const assembly = super.synth(options);
        const typesToFail = new Set([
            this.failOnError && CDK_NAG_MESSAGE_TYPES.ERROR,
            this.failOnWarning && CDK_NAG_MESSAGE_TYPES.WARNING,
        ].filter((t) => t));
        if (this._nagResults.find((r) => r.messages.find((m) => typesToFail.has(m.messageType)))) {
            throw new Error(JSON.stringify(this._nagResults, undefined, 2));
        }
        return assembly;
    }
    addNagResult(result) {
        this._nagResults.push(result);
    }
    /**
     * Returns a list of NagResult.
     *
     * Note: app.synth() must be called before this to retrieve results.
     */
    nagResults() {
        return this._nagResults;
    }
}
exports.PDKNagApp = PDKNagApp;
_a = JSII_RTTI_SYMBOL_1;
PDKNagApp[_a] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNagApp", version: "0.11.2" };
class PDKNagAspect extends cdk_nag_1.AwsSolutionsChecks {
    constructor(app) {
        super({
            verbose: true,
            reports: true,
        });
        this.app = app;
    }
    visit(node) {
        super.visit(node);
        const results = node.node.metadata.filter((m) => CDK_NAG_MESSAGE_TYPES_SET.has(m.type));
        results.length > 0 &&
            this.app.addNagResult({
                resource: node.node.path,
                messages: results.map((m) => ({
                    messageDescription: m.data,
                    messageType: m.type,
                })),
            });
    }
}
/**
 * Helper for create a Nag Enabled App.
 */
class PDKNag {
    /**
     * Returns an instance of an App with Nag enabled.
     *
     * @param props props to initialize the app with.
     */
    static app(props) {
        const app = new PDKNagApp(props);
        aws_cdk_lib_1.Aspects.of(app).add(new PDKNagAspect(app));
        return app;
    }
    /**
     * Wrapper around NagSuppressions which does not throw.
     *
     * @param stack stack instance
     * @param path resource path
     * @param suppressions list of suppressions to apply.
     * @param applyToChildren whether to apply to children.
     */
    static addResourceSuppressionsByPathNoThrow(stack, path, suppressions, applyToChildren = false) {
        try {
            cdk_nag_1.NagSuppressions.addResourceSuppressionsByPath(stack, path, suppressions, applyToChildren);
        }
        catch (e) {
            // Do Nothing
        }
    }
    /**
     * Returns a prefix comprising of a delimited set of Stack Ids.
     *
     * For example: StackA/NestedStackB/
     *
     * @param stack stack instance.
     */
    static getStackPrefix(stack) {
        if (stack.nested) {
            return `${PDKNag.getStackPrefix(stack.nestedStackParent)}${stack.node.id}/`;
        }
        else {
            const stageName = aws_cdk_lib_1.Stage.of(stack)?.stageName;
            const stagePrefix = stageName && `${stageName}-`;
            let stackName = stack.stackName;
            stackName =
                stagePrefix && stackName.startsWith(stagePrefix)
                    ? `${stageName}/${stackName.slice(stagePrefix.length)}`
                    : stackName;
            return `${stackName}/`;
        }
    }
    /**
     * Returns a stack partition regex.
     *
     * @param stack stack instance.
     */
    static getStackPartitionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackPartitionRegex(stack.nestedStackParent);
        }
        else {
            return stack.partition.startsWith("${Token")
                ? "<AWS::Partition>"
                : `(<AWS::Partition>|${stack.partition})`;
        }
    }
    /**
     * Returns a stack region regex.
     *
     * @param stack stack instance.
     */
    static getStackRegionRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackRegionRegex(stack.nestedStackParent);
        }
        else {
            return stack.region.startsWith("${Token")
                ? "<AWS::Region>"
                : `(<AWS::Region>|${stack.region})`;
        }
    }
    /**
     * Returns a stack account regex.
     *
     * @param stack stack instance.
     */
    static getStackAccountRegex(stack) {
        if (stack.nested) {
            return PDKNag.getStackAccountRegex(stack.nestedStackParent);
        }
        else {
            return stack.account.startsWith("${Token")
                ? "<AWS::AccountId>"
                : `(<AWS::AccountId>|${stack.account})`;
        }
    }
}
exports.PDKNag = PDKNag;
_b = JSII_RTTI_SYMBOL_1;
PDKNag[_b] = { fqn: "aws-prototyping-sdk.pdk_nag.PDKNag", version: "0.11.2" };
//# sourceMappingURL=data:application/json;base64,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