"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayLambdaApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const pdk_nag_1 = require("@aws-prototyping-sdk/pdk-nag");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_apigateway_1 = require("aws-cdk-lib/aws-apigateway");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const cdk_nag_1 = require("cdk-nag");
const constructs_1 = require("constructs");
const authorizers_1 = require("./authorizers");
const prepare_spec_1 = require("./prepare-spec-event-handler/prepare-spec");
const api_gateway_auth_1 = require("./spec/api-gateway-auth");
const api_gateway_integrations_1 = require("./spec/api-gateway-integrations");
const utils_1 = require("./spec/utils");
const open_api_gateway_web_acl_1 = require("./waf/open-api-gateway-web-acl");
/**
 * A construct for creating an api gateway api based on the definition in the OpenAPI spec.
 */
class OpenApiGatewayLambdaApi extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { integrations, spec, specPath, operationLookup, defaultAuthorizer, corsOptions, ...options } = props;
        // Upload the spec to s3 as an asset
        const inputSpecAsset = new aws_s3_assets_1.Asset(this, "InputSpec", {
            path: specPath,
        });
        // We'll output the prepared spec in the same asset bucket
        const preparedSpecOutputKeyPrefix = `${inputSpecAsset.s3ObjectKey}-prepared`;
        const stack = aws_cdk_lib_1.Stack.of(this);
        const prepareSpecLambdaName = `${pdk_nag_1.PDKNag.getStackPrefix(stack)
            .split("/")
            .join("-")}PrepareSpec`;
        const prepareSpecRole = new aws_iam_1.Role(this, "PrepareSpecRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${prepareSpecLambdaName}:*`,
                            ],
                        }),
                    ],
                }),
                s3: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:getObject"],
                            resources: [
                                inputSpecAsset.bucket.arnForObjects(inputSpecAsset.s3ObjectKey),
                            ],
                        }),
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: ["s3:putObject"],
                            resources: [
                                // The output file will include a hash of the prepared spec, which is not known until deploy time since
                                // tokens must be resolved
                                inputSpecAsset.bucket.arnForObjects(`${preparedSpecOutputKeyPrefix}/*`),
                            ],
                        }),
                    ],
                }),
            },
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(prepareSpecRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
                appliesTo: [
                    {
                        regex: `/^Resource::arn:aws:logs:${pdk_nag_1.PDKNag.getStackRegionRegex(stack)}:${pdk_nag_1.PDKNag.getStackAccountRegex(stack)}:log-group:/aws/lambda/${prepareSpecLambdaName}:\*/g`,
                    },
                ],
            },
        ], true);
        // Create a custom resource for preparing the spec for deployment (adding integrations, authorizers, etc)
        const prepareSpec = new aws_lambda_1.Function(this, "PrepareSpec", {
            handler: "index.handler",
            runtime: aws_lambda_1.Runtime.NODEJS_16_X,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../../lib/construct/prepare-spec-event-handler")),
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            role: prepareSpecRole,
            functionName: prepareSpecLambdaName,
        });
        const providerFunctionName = `${prepareSpecLambdaName}-Provider`;
        const providerRole = new aws_iam_1.Role(this, "PrepareSpecProviderRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            inlinePolicies: {
                logs: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            effect: aws_iam_1.Effect.ALLOW,
                            actions: [
                                "logs:CreateLogGroup",
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                            ],
                            resources: [
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}`,
                                `arn:aws:logs:${stack.region}:${stack.account}:log-group:/aws/lambda/${providerFunctionName}:*`,
                            ],
                        }),
                    ],
                }),
            },
        });
        const provider = new custom_resources_1.Provider(this, "PrepareSpecProvider", {
            onEventHandler: prepareSpec,
            role: providerRole,
            providerFunctionName,
        });
        cdk_nag_1.NagSuppressions.addResourceSuppressions(providerRole, [
            {
                id: "AwsSolutions-IAM5",
                reason: "Cloudwatch resources have been scoped down to the LogGroup level, however * is still needed as stream names are created just in time.",
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(provider, [
            {
                id: "AwsSolutions-L1",
                reason: "Latest runtime cannot be configured. CDK will need to upgrade the Provider construct accordingly.",
            },
        ], true);
        const prepareSpecOptions = {
            integrations: Object.fromEntries(Object.entries(integrations).map(([operation, integration]) => [
                operation,
                {
                    functionInvocationUri: utils_1.functionInvocationUri(this, integration.function),
                    methodAuthorizer: api_gateway_auth_1.serializeAsAuthorizerReference(integration.authorizer || defaultAuthorizer || authorizers_1.Authorizers.none()),
                },
            ])),
            securitySchemes: api_gateway_auth_1.prepareSecuritySchemes(this, integrations, defaultAuthorizer),
            corsOptions: corsOptions && {
                allowHeaders: corsOptions.allowHeaders || aws_apigateway_1.Cors.DEFAULT_HEADERS,
                allowMethods: corsOptions.allowMethods || aws_apigateway_1.Cors.ALL_METHODS,
                allowOrigins: corsOptions.allowOrigins,
                statusCode: corsOptions.statusCode || 204,
            },
            operationLookup,
        };
        // Spec preparation will happen in a custom resource lambda so that references to lambda integrations etc can be
        // resolved. However, we also prepare inline to perform some additional validation at synth time.
        const preparedSpec = prepare_spec_1.prepareApiSpec(spec, prepareSpecOptions);
        const prepareApiSpecCustomResourceProperties = {
            inputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: inputSpecAsset.s3ObjectKey,
            },
            outputSpecLocation: {
                bucket: inputSpecAsset.bucket.bucketName,
                key: preparedSpecOutputKeyPrefix,
            },
            ...prepareSpecOptions,
        };
        const prepareSpecCustomResource = new aws_cdk_lib_1.CustomResource(this, "PrepareSpecCustomResource", {
            serviceToken: provider.serviceToken,
            properties: prepareApiSpecCustomResourceProperties,
        });
        // Create the api gateway resources from the spec, augmenting the spec with the properties specific to api gateway
        // such as integrations or auth types
        this.api = new aws_apigateway_1.SpecRestApi(this, id, {
            apiDefinition: aws_apigateway_1.ApiDefinition.fromBucket(inputSpecAsset.bucket, prepareSpecCustomResource.getAttString("outputSpecKey")),
            deployOptions: {
                accessLogDestination: new aws_apigateway_1.LogGroupLogDestination(new aws_logs_1.LogGroup(scope, `AccessLogs`)),
                accessLogFormat: aws_apigateway_1.AccessLogFormat.clf(),
                loggingLevel: aws_apigateway_1.MethodLoggingLevel.INFO,
            },
            ...options,
        });
        this.api.node.addDependency(prepareSpecCustomResource);
        // While the api will be updated when the output path from the custom resource changes, CDK still needs to know when
        // to redeploy the api. This is achieved by including a hash of the spec in the logical id (internalised in the
        // addToLogicalId method since this is how changes of individual resources/methods etc trigger redeployments in CDK)
        this.api.latestDeployment?.addToLogicalId(preparedSpec);
        // Grant API Gateway permission to invoke each lambda which implements an integration or custom authorizer
        api_gateway_integrations_1.getLabelledFunctions(props).forEach(({ label, function: lambda }) => {
            new aws_lambda_1.CfnPermission(this, `LambdaPermission-${label}`, {
                action: "lambda:InvokeFunction",
                principal: "apigateway.amazonaws.com",
                functionName: lambda.functionArn,
                sourceArn: stack.formatArn({
                    service: "execute-api",
                    resource: this.api.restApiId,
                    resourceName: "*/*",
                }),
            });
        });
        // Create and associate the web acl if not disabled
        if (!props.webAclOptions?.disable) {
            const acl = new open_api_gateway_web_acl_1.OpenApiGatewayWebAcl(this, `${id}-Acl`, {
                ...props.webAclOptions,
                apiDeploymentStageArn: this.api.deploymentStage.stageArn,
            });
            this.webAcl = acl.webAcl;
            this.ipSet = acl.ipSet;
            this.webAclAssociation = acl.webAclAssociation;
        }
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-IAM4",
                reason: "Cloudwatch Role requires access to create/read groups at the root level.",
                appliesTo: [
                    {
                        regex: `/^Policy::arn:${pdk_nag_1.PDKNag.getStackPartitionRegex(stack)}:iam::aws:policy/service-role/AmazonAPIGatewayPushToCloudWatchLogs$/g`,
                    },
                ],
            },
        ], true);
        cdk_nag_1.NagSuppressions.addResourceSuppressions(this, [
            {
                id: "AwsSolutions-APIG2",
                reason: "This construct implements fine grained validation via OpenApi.",
            },
        ], true);
    }
}
exports.OpenApiGatewayLambdaApi = OpenApiGatewayLambdaApi;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayLambdaApi[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayLambdaApi", version: "0.9.3" };
//# sourceMappingURL=data:application/json;base64,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