"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayJavaProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const java_1 = require("projen/lib/java");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const java_2 = require("./samples/java");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "software.aws.awsprototypingsdk/open-api-gateway@^0";
/**
 * Synthesizes a Java Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-java
 */
class OpenApiGatewayJavaProject extends java_1.JavaProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "software.constructs/constructs@^10",
            "software.amazon.awscdk/aws-cdk-lib@^2",
            "io.github.cdklabs/cdknag@^2",
            "org.projectlombok/lombok@^1",
            "com.fasterxml.jackson.core/jackson-databind@^2",
            "io.github.cdklabs/projen@^0",
        ].forEach((dep) => this.addDependency(dep));
        // Remove the projen test dependency since otherwise it takes precedence, causing projen to be unavailable at synth time
        this.deps.removeDependency("io.github.cdklabs/projen", projen_1.DependencyType.TEST);
        // Synthesize the openapi spec early since it's used by the generated java client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join("src", this.specDir),
            // Write the parsed spec to the resources directory so that it can be packaged into the jar
            parsedSpecOutdir: path.join(...this.specDir.split("/").map(() => ".."), "main", "resources", this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the java client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.JAVA);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.JAVA,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: "APJ Cope",
                authorEmail: "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedJavaClient = this.generatedClients[languages_1.ClientLanguage.JAVA];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedJavaClient.synth();
        // Add a dependency on the generated java client
        this.addDependency(`${this.generatedJavaClient.pom.groupId}/${this.generatedJavaClient.pom.artifactId}@${this.generatedJavaClient.pom.version}`);
        // Add a dependency on the generated java client repository
        this.pom.addRepository({
            url: `file://./${this.generatedCodeDir}/${languages_1.ClientLanguage.JAVA}/dist/java`,
            id: "generated-java-api-client",
        });
        if (this.hasParent) {
            // Since the generated java client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
            if ("addImplicitDependency" in this.parent) {
                // If we're within a monorepo, add an implicit dependency to ensure the generated java client is built first
                this.parent.addImplicitDependency(this, this.generatedJavaClient);
            }
        }
        // We build this project as a super jar so that it can be deployed as a lambda. This isn't ideal for jar size and
        // so is not the recommended approach, however is makes it possible for a "one-click" way to get started with a
        // full hello world api. Included in the generated "SampleApi.java" is a comment encouraging users to read the
        // README and to move to defining a separate package for lambda handlers.
        this.pom.addPlugin("org.apache.maven.plugins/maven-shade-plugin@3.3.0", {
            configuration: {
                createDependencyReducedPom: false,
                // Name is the same as the regular maven build jar, which ensures the api jar path resolves to the super jar
                // for deployment as a lambda.
                finalName: `${this.name}-${options.version}`,
            },
            executions: [
                {
                    id: "shade-task",
                    phase: "package",
                    goals: ["shade"],
                },
            ],
        });
        const javaClientPackage = `${this.generatedJavaClient.pom.groupId}.${this.generatedJavaClient.name}.client`;
        // Generate the sample source code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            sampleCode: options.sample ?? true,
            apiSrcDir: this.apiSrcDir,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            javaClientPackage,
        };
        new projen_1.SampleDir(this, path.join("src", "main", "java", this.apiSrcDir), {
            files: java_2.getJavaSampleSource(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
}
exports.OpenApiGatewayJavaProject = OpenApiGatewayJavaProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayJavaProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayJavaProject", version: "0.9.3" };
//# sourceMappingURL=data:application/json;base64,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