"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CatalogClient = void 0;
const aws_lambda_shared_1 = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
/**
 * A client for working with the catalog.
 */
class CatalogClient {
    constructor() {
        this.bucketName = env_lambda_shared_1.requireEnv('CATALOG_BUCKET_NAME');
        this.objectKey = env_lambda_shared_1.requireEnv('CATALOG_OBJECT_KEY');
        this.s3 = aws_lambda_shared_1.s3();
    }
    /**
     * Creates a new client for accessing the catalog.
     */
    static async newClient() {
        const client = new CatalogClient();
        await client.init();
        return client;
    }
    /**
     * Downloads the catalog object and stores it in memory.
     */
    async init() {
        if (this._packages) {
            throw new Error('init() cannot be called twice');
        }
        this._packages = [];
        const params = {
            Bucket: this.bucketName,
            Key: this.objectKey,
        };
        let body;
        try {
            const data = await this.s3.getObject(params).promise();
            body = data.Body;
        }
        catch (e) {
            throw new Error(`No catalog was found at ${this.bucketName}/${this.objectKey}`);
        }
        if (!body) {
            throw new Error(`Catalog body is empty at ${this.bucketName}/${this.objectKey}`);
        }
        const contents = body.toString('utf-8');
        try {
            const data = JSON.parse(contents);
            if (typeof (data) != 'object') {
                throw new Error('Invalid format in catalog file. Expecting a map');
            }
            this._packages = data.packages;
        }
        catch (e) {
            throw new Error(`Unable to parse catalog file ${this.bucketName}/${this.objectKey}: ${e}`);
        }
    }
    /**
     * Returns a copy of the catalog list.
     */
    get packages() {
        if (!this._packages) {
            throw new Error('CatalogClient must be initialized');
        }
        return [...this._packages];
    }
}
exports.CatalogClient = CatalogClient;
//# sourceMappingURL=data:application/json;base64,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