"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CatalogBuilder = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const catalog_builder_1 = require("./catalog-builder");
/**
 * Builds or re-builds the `catalog.json` object in the designated bucket.
 */
class CatalogBuilder extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const handler = new catalog_builder_1.CatalogBuilder(this, 'Default', {
            description: `Creates the catalog.json object in ${props.bucket.bucketName}`,
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                AWS_EMF_ENVIRONMENT: 'Local',
            },
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
        });
        this.function = handler;
        // This function may invoke itself in case it needs to continue it's work in
        // a "child" invocation. We must hence allow it to invoke itself. We cannot
        // use grantInvoke as this would (naturally) cause a circular reference
        // (Function -> Role -> Function).
        handler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [core_1.Stack.of(this).formatArn({
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                    service: 'lambda',
                    resource: 'function',
                    resourceName: '*',
                })],
        }));
        // allow the catalog builder to use the client.
        props.denyList.grantRead(handler);
        props.bucket.grantReadWrite(this.function);
    }
}
exports.CatalogBuilder = CatalogBuilder;
//# sourceMappingURL=data:application/json;base64,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