"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const url_1 = require("url");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const tags_1 = require("../shared/tags");
const tarball_lambda_shared_1 = require("../shared/tarball.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const STATE_MACHINE_ARN = env_lambda_shared_1.requireEnv('STATE_MACHINE_ARN');
    const PACKAGE_LINKS = env_lambda_shared_1.requireEnv('PACKAGE_LINKS');
    const PACKAGE_TAGS = env_lambda_shared_1.requireEnv('PACKAGE_TAGS');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws
            .s3()
            .getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        })
            .promise();
        const { integrity: integrityCheck } = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        let dotJsii;
        let packageJson;
        let licenseText;
        try {
            ({ dotJsii, packageJson, licenseText } = await tarball_lambda_shared_1.extractObjects(Buffer.from(tarball.Body), {
                dotJsii: { path: 'package/.jsii', required: true },
                packageJson: { path: 'package/package.json', required: true },
                licenseText: { filter: isLicenseFile },
            }));
        }
        catch (err) {
            console.error(`Invalid tarball content: ${err}`);
            metrics.putMetric("InvalidTarball" /* INVALID_TARBALL */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        let constructFramework;
        let packageLicense;
        let packageName;
        let packageVersion;
        try {
            const assembly = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
            constructFramework = detectConstructFramework(assembly);
            const { license, name, version } = assembly;
            packageLicense = license;
            packageName = name;
            packageVersion = version;
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 0, aws_embedded_metrics_1.Unit.Count);
        }
        catch (ex) {
            console.error(`Package does not contain a valid assembly -- ignoring: ${ex}`);
            metrics.putMetric("InvalidAssembly" /* INVALID_ASSEMBLY */, 1, aws_embedded_metrics_1.Unit.Count);
            return;
        }
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const packageJsonObj = JSON.parse(packageJson.toString('utf-8'));
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense, constructHub, } = packageJsonObj;
        if (packageJsonName !== packageName ||
            packageJsonVersion !== packageVersion ||
            packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        // Add custom links content to metdata for display on the frontend
        const allowedLinks = JSON.parse(PACKAGE_LINKS);
        const packageLinks = allowedLinks.reduce((accum, { configKey, allowedDomains }) => {
            const pkgValue = constructHub === null || constructHub === void 0 ? void 0 : constructHub.packageLinks[configKey];
            if (!pkgValue) {
                return accum;
            }
            // check if value is in allowed domains list
            const url = new url_1.URL(pkgValue);
            if ((allowedDomains === null || allowedDomains === void 0 ? void 0 : allowedDomains.length) && !allowedDomains.includes(url.host)) {
                return accum;
            }
            // if no allow list is provided
            return { ...accum, [configKey]: pkgValue };
        }, {});
        // Add computed tags to metadata
        const packageTagsConfig = JSON.parse(PACKAGE_TAGS);
        const packageTags = packageTagsConfig.reduce((accum, tagConfig) => {
            const { condition, ...tagData } = tagConfig;
            if (tags_1.isTagApplicable(condition, packageJsonObj)) {
                return [...accum, tagData];
            }
            return accum;
        }, []);
        const metadata = {
            constructFramework,
            date: payload.time,
            licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8'),
            packageLinks,
            packageTags,
        };
        const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
        console.log(`Writing assembly at ${assemblyKey}`);
        console.log(`Writing package at  ${packageKey}`);
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                CacheControl: 'public, max-age=86400, must-revalidate, s-maxage=300, proxy-revalidate',
                ContentType: 'application/octet-stream',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
            aws
                .s3()
                .putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            })
                .promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws
            .s3()
            .putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            CacheControl: 'public, max-age: 86400, must-revalidate, s-maxage=300, proxy-revalidate',
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        })
            .promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        const sfn = await aws
            .stepFunctions()
            .startExecution({
            input: JSON.stringify(created),
            name: sfnExecutionNameFromParts(packageName, `v${packageVersion}`, context.awsRequestId),
            stateMachineArn: STATE_MACHINE_ARN,
        })
            .promise();
        console.log(`Started StateMachine execution: ${sfn.executionArn}`);
        result.push(sfn.executionArn);
    }
    return result;
});
/**
 * Determines the Construct framework used by the provided assembly.
 *
 * @param assembly the assembly for which a construct framework should be
 *                 identified.
 *
 * @returns a construct framework if one could be identified.
 */
function detectConstructFramework(assembly) {
    var _a;
    let name;
    let nameAmbiguous = false;
    let majorVersion;
    let majorVersionAmbiguous = false;
    detectConstructFrameworkPackage(assembly.name, assembly.version);
    for (const depName of Object.keys((_a = assembly.dependencyClosure) !== null && _a !== void 0 ? _a : {})) {
        detectConstructFrameworkPackage(depName);
        if (nameAmbiguous) {
            return undefined;
        }
    }
    return name && { name, majorVersion: majorVersionAmbiguous ? undefined : majorVersion };
    function detectConstructFrameworkPackage(packageName, versionRange) {
        var _a, _b;
        if (versionRange === void 0) { versionRange = (_a = assembly.dependencies) === null || _a === void 0 ? void 0 : _a[packageName]; }
        if (packageName.startsWith('@aws-cdk/') || packageName === 'aws-cdk-lib' || packageName === 'monocdk') {
            if (name && name !== "aws-cdk" /* AWS_CDK */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "aws-cdk" /* AWS_CDK */;
        }
        else if (packageName === 'cdktf' || packageName.startsWith('@cdktf/')) {
            if (name && name !== "cdktf" /* CDKTF */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdktf" /* CDKTF */;
        }
        else if (packageName === 'cdk8s' || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(packageName)) {
            if (name && name !== "cdk8s" /* CDK8S */) {
                // Identified multiple candidates, so returning ambiguous...
                nameAmbiguous = true;
                return;
            }
            name = "cdk8s" /* CDK8S */;
        }
        else {
            return;
        }
        if (versionRange) {
            const major = (_b = semver_1.minVersion(versionRange)) === null || _b === void 0 ? void 0 : _b.major;
            if (majorVersion != null && majorVersion !== major) {
                // Identified multiple candidates, so this is ambiguous...
                majorVersionAmbiguous = true;
            }
            majorVersion = major;
        }
        return;
    }
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return (possibleExtensions.has(ext.toLowerCase()) &&
        path_1.basename(fileName, ext).toUpperCase() === 'LICENSE');
}
/**
 * Creates a StepFunction execution request name based on the provided parts.
 * The result is guaranteed to be 80 characters or less and to contain only
 * characters that are valid for a StepFunction execution request name for which
 * CloudWatch Logging can be enabled. The resulting name is very likely to
 * be unique for a given input.
 */
function sfnExecutionNameFromParts(first, ...rest) {
    const parts = [first, ...rest];
    const name = parts
        .map((part) => part.replace(/[^a-z0-9_-]+/gi, '_'))
        .join('_')
        .replace(/^_/g, '')
        .replace(/_{2,}/g, '_');
    if (name.length <= 80) {
        return name;
    }
    const suffix = crypto_1.createHash('sha256')
        // The hash is computed based on input arguments, to maximize unicity
        .update(parts.join('_'))
        .digest('hex')
        .substring(0, 6);
    return `${name.substring(0, 80 - suffix.length - 1)}_${suffix}`;
}
//# sourceMappingURL=data:application/json;base64,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