"use strict";
/*
 * IMPORTANT: This file is used both by bundled lambda functions and by
 * constructs. This means it should not directly or transitively import anything
 * not part of the package's `dependencies`, such as the `aws-sdk`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.STORAGE_KEY_FORMAT_REGEX = exports.CATALOG_KEY = exports.NOT_SUPPORTED_SUFFIX = exports.docsKeySuffix = exports.getObjectKeys = exports.getObjectKeyPrefix = exports.DOCS_KEY_SUFFIX_ANY = exports.DOCS_KEY_SUFFIX_JAVA = exports.DOCS_KEY_SUFFIX_PYTHON = exports.DOCS_KEY_SUFFIX_TYPESCRIPT = exports.ASSEMBLY_KEY_SUFFIX = exports.METADATA_KEY_SUFFIX = exports.PACKAGE_KEY_SUFFIX = exports.STORAGE_KEY_PREFIX = void 0;
const language_1 = require("./language");
/**
 * Key prefix for the package data storage.
 */
exports.STORAGE_KEY_PREFIX = 'data/';
/**
 * Key suffix for storing npm package bundles.
 */
exports.PACKAGE_KEY_SUFFIX = '/package.tgz';
/**
 * Key suffix for storing npm package metadata.
 */
exports.METADATA_KEY_SUFFIX = '/metadata.json';
/**
 * The key suffix for (TypeScript) assembly files
 */
exports.ASSEMBLY_KEY_SUFFIX = '/assembly.json';
/**
 * The key suffix for a TypeScript doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_TYPESCRIPT = docsKeySuffix(language_1.DocumentationLanguage.TYPESCRIPT);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_PYTHON = docsKeySuffix(language_1.DocumentationLanguage.PYTHON);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_JAVA = docsKeySuffix(language_1.DocumentationLanguage.JAVA);
/**
 * The key suffix matching any documentation artifact.
 */
exports.DOCS_KEY_SUFFIX_ANY = docsKeySuffix('*');
/**
 * Return the S3 object key prefix for a specific package name and optionally a
 * version. Note that the prefix does not end with a "/" so you will likely want
 * to add that if you want to match a specific set of objects.
 */
function getObjectKeyPrefix(packageName, packageVersion) {
    let key = `${exports.STORAGE_KEY_PREFIX}${packageName}`;
    if (packageVersion) {
        key += `/v${packageVersion}`;
    }
    return key;
}
exports.getObjectKeyPrefix = getObjectKeyPrefix;
/**
 * Resolves the set of S3 object keys use for a specific package/version.
 */
function getObjectKeys(packageName, packageVersion) {
    const prefix = getObjectKeyPrefix(packageName, packageVersion);
    return {
        assemblyKey: `${prefix}${exports.ASSEMBLY_KEY_SUFFIX}`,
        packageKey: `${prefix}${exports.PACKAGE_KEY_SUFFIX}`,
        metadataKey: `${prefix}${exports.METADATA_KEY_SUFFIX}`,
    };
}
exports.getObjectKeys = getObjectKeys;
/**
 * The key suffix for documentation artifacts by language and submodule.
 */
function docsKeySuffix(lang, submodule) {
    return `/docs-${submodule ? `${submodule}-` : ''}${lang}.md`;
}
exports.docsKeySuffix = docsKeySuffix;
/**
 * Key suffix for beacon files when a particular feature is not supported for
 * the particular package (i.e: Python docs for a package that does not have a
 * Python target configured).
 */
exports.NOT_SUPPORTED_SUFFIX = '.not-supported';
/**
 * The key for the catalog document.
 */
exports.CATALOG_KEY = 'catalog.json';
/**
 * A regular expression that can be used to parse out a storage key.
 */
exports.STORAGE_KEY_FORMAT_REGEX = new RegExp(`^${exports.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                        ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
//# sourceMappingURL=data:application/json;base64,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