"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const util_1 = require("projen/lib/util");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const typescript_2 = require("./samples/typescript");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "@aws-prototyping-sdk/open-api-gateway";
/**
 * Synthesizes a Typescript Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-ts
 */
class OpenApiGatewayTsProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            sampleCode: false,
            tsconfig: {
                compilerOptions: {
                    lib: ["dom", "es2019"],
                },
            },
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Allow json files to be imported (for importing the parsed spec)
        this.tsconfig?.addInclude(`${this.srcdir}/**/*.json`);
        // Set to private since this either uses workspaces or has file dependencies
        this.package.addField("private", true);
        // Generated project should have a dependency on this project, in order to run the generation scripts
        this.addDeps(OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib");
        // Synthesize the openapi spec early since it's used by the generated typescript client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.srcdir, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        new projen_1.TextFile(this, path.join(this.generatedCodeDir, "README.md"), {
            lines: [
                "## Generated Clients",
                "",
                "This directory contains generated client code based on your OpenAPI Specification file (spec.yaml).",
                "",
                "Like other `projen` managed files, this directory should be checked in to source control, but should not be edited manually.",
            ],
            readonly: true,
        });
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the typescript client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.TYPESCRIPT);
        this.generatedClients = generate_1.generateClientProjects(clientLanguages, {
            parent: this.hasParent ? options.parent : this,
            rootProjectHasParent: this.hasParent,
            parentPackageName: this.package.packageName,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: options.defaultReleaseBranch,
                packageManager: options.packageManager,
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
            },
        });
        this.generatedTypescriptClient = this.generatedClients[languages_1.ClientLanguage.TYPESCRIPT];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedTypescriptClient.synth();
        const typescriptCodeGenDir = path.relative(this.outdir, this.generatedTypescriptClient.outdir);
        if (this.hasParent) {
            // When we have a parent project, we can add a dependency on the generated typescript client since it's part of
            // the monorepo. Since this project will be synthesized before the parent monorepo, the generated typescript
            // client won't be visible for the first install in this project's post synthesize step, so we use a local
            // workspace as a workaround.
            if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
                this.pnpmWorkspace = new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                    readonly: true,
                    obj: {
                        packages: [typescriptCodeGenDir],
                    },
                });
            }
            else {
                this.package.addField("workspaces", {
                    packages: [typescriptCodeGenDir],
                });
            }
            // Add the dependency
            this.addDeps(this.generatedTypescriptClient.package.packageName);
            // Since the generated typescript client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        else {
            // Add a file dependency on the generated typescript client
            this.addDeps(`${this.generatedTypescriptClient.package.packageName}@file:./${typescriptCodeGenDir}`);
            // Linting runs post synthesize before the typescript client has been built
            this.eslint?.addRules({
                "import/no-unresolved": [
                    2,
                    { ignore: [this.generatedTypescriptClient.package.packageName] },
                ],
            });
        }
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            typescriptClientPackageName: this.generatedTypescriptClient.package.packageName,
            sampleCode: options.sampleCode,
            apiSrcDir: path.join(this.srcdir, this.apiSrcDir),
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
        };
        new projen_1.SampleFile(this, path.join(this.srcdir, "index.ts"), {
            contents: `export * from "./${this.apiSrcDir}";`,
        });
        new projen_1.SampleDir(this, path.join(this.srcdir, this.apiSrcDir), {
            files: typescript_2.getTypescriptSampleSource(sampleOptions),
        });
        new projen_1.SampleDir(this, this.testdir, {
            files: typescript_2.getTypescriptSampleTests(sampleOptions),
        });
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // When no parent is passed, link the generated client as a prebuild step to ensure the latest built generated
        // client is reflected in this package's node modules.
        // Note that it's up to the user to manage building the generated client first.
        !this.hasParent && this.executeLinkNativeClientCommands();
        super.postSynthesize();
    }
    /**
     * Executes commands to link the native (ie typescript) client such that updates to the generated client are reflected
     * in this project's node_modules (when not managed by a monorepo)
     * @private
     */
    executeLinkNativeClientCommands() {
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.NPM:
            case javascript_1.NodePackageManager.YARN:
                util_1.exec(`${this.package.packageManager} link`, {
                    cwd: this.generatedTypescriptClient.outdir,
                });
                util_1.exec(`${this.package.packageManager} link ${this.generatedTypescriptClient.package.packageName}`, {
                    cwd: this.outdir,
                });
                break;
            case javascript_1.NodePackageManager.PNPM:
                util_1.exec(`${this.package.packageManager} link ./${path.relative(this.outdir, this.generatedTypescriptClient.outdir)}`, {
                    cwd: this.outdir,
                });
                break;
            default:
                throw new Error(`Unsupported package manager ${this.package.packageManager}`);
        }
    }
}
exports.OpenApiGatewayTsProject = OpenApiGatewayTsProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayTsProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayTsProject", version: "0.2.3" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3Blbi1hcGktZ2F0ZXdheS10cy1wcm9qZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL3Byb2plY3Qvb3Blbi1hcGktZ2F0ZXdheS10cy1wcm9qZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7Ozs7Ozs7Ozs7Ozs7O3dIQWN3SDtBQUV4SCw2QkFBNkI7QUFDN0IsbUNBQTRFO0FBQzVFLHNEQUEyRDtBQUMzRCxzREFHK0I7QUFDL0IsMENBQXVDO0FBQ3ZDLGlEQUE0RDtBQUU1RCwyQ0FBNkM7QUFDN0MscURBSThCO0FBQzlCLHdFQUFrRTtBQUdsRSxNQUFNLGdDQUFnQyxHQUNwQyx1Q0FBdUMsQ0FBQztBQVMxQzs7Ozs7R0FLRztBQUNILE1BQWEsdUJBQXdCLFNBQVEsOEJBQWlCO0lBdUM1RCxZQUFZLE9BQXVDO1FBQ2pELEtBQUssQ0FBQztZQUNKLEdBQUcsT0FBTztZQUNWLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRTtnQkFDUixlQUFlLEVBQUU7b0JBQ2YsR0FBRyxFQUFFLENBQUMsS0FBSyxFQUFFLFFBQVEsQ0FBQztpQkFDdkI7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksT0FBTyxDQUFDLFFBQVEsRUFBRTtZQUNwQixJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQzlDLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLENBQUM7U0FDckQ7YUFBTTtZQUNMLElBQUksQ0FBQyxPQUFPLEdBQUcsTUFBTSxDQUFDO1lBQ3RCLElBQUksQ0FBQyxZQUFZLEdBQUcsV0FBVyxDQUFDO1NBQ2pDO1FBQ0QsSUFBSSxDQUFDLGdCQUFnQixHQUFHLE9BQU8sQ0FBQyxnQkFBZ0IsSUFBSSxXQUFXLENBQUM7UUFDaEUsSUFBSSxDQUFDLFNBQVMsR0FBRyxPQUFPLENBQUMsU0FBUyxJQUFJLEtBQUssQ0FBQztRQUU1QyxrRUFBa0U7UUFDbEUsSUFBSSxDQUFDLFFBQVEsRUFBRSxVQUFVLENBQUMsR0FBRyxJQUFJLENBQUMsTUFBTSxZQUFZLENBQUMsQ0FBQztRQUV0RCw0RUFBNEU7UUFDNUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLElBQUksQ0FBQyxDQUFDO1FBRXZDLHFHQUFxRztRQUNyRyxJQUFJLENBQUMsT0FBTyxDQUFDLGdDQUFnQyxFQUFFLFlBQVksRUFBRSxhQUFhLENBQUMsQ0FBQztRQUU1RSxvSEFBb0g7UUFDcEgsTUFBTSxJQUFJLEdBQUcsSUFBSSwwQ0FBa0IsQ0FBQztZQUNsQyxJQUFJLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxPQUFPO1lBQ3pCLE1BQU0sRUFBRSxJQUFJO1lBQ1osTUFBTSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxJQUFJLENBQUMsT0FBTyxDQUFDO1lBQzVDLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMvQixrQkFBa0IsRUFBRSxPQUFPLENBQUMsa0JBQWtCO1NBQy9DLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUViLElBQUksaUJBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsV0FBVyxDQUFDLEVBQUU7WUFDaEUsS0FBSyxFQUFFO2dCQUNMLHNCQUFzQjtnQkFDdEIsRUFBRTtnQkFDRixxR0FBcUc7Z0JBQ3JHLEVBQUU7Z0JBQ0YsOEhBQThIO2FBQy9IO1lBQ0QsUUFBUSxFQUFFLElBQUk7U0FDZixDQUFDLENBQUM7UUFFSCw2RkFBNkY7UUFDN0YsSUFBSSxDQUFDLFNBQVMsR0FBRyxDQUFDLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUNsQyxNQUFNLGdDQUFnQyxHQUFHLElBQUksQ0FBQyxTQUFTO1lBQ3JELENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxNQUFPLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDO1lBQ25ELENBQUMsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUM7UUFFMUIsZ0hBQWdIO1FBQ2hILG1DQUFtQztRQUNuQyxNQUFNLGVBQWUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxPQUFPLENBQUMsZUFBZSxDQUFDLENBQUM7UUFDekQsZUFBZSxDQUFDLEdBQUcsQ0FBQywwQkFBYyxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBRS9DLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxpQ0FBc0IsQ0FBQyxlQUFlLEVBQUU7WUFDOUQsTUFBTSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUk7WUFDOUMsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLFNBQVM7WUFDcEMsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQzNDLGdCQUFnQixFQUFFLGdDQUFnQztZQUNsRCxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7WUFDbkMsaUJBQWlCLEVBQUU7Z0JBQ2pCLG9CQUFvQixFQUFFLE9BQU8sQ0FBQyxvQkFBb0I7Z0JBQ2xELGNBQWMsRUFBRSxPQUFPLENBQUMsY0FBYztnQkFDdEMsR0FBRyxPQUFPLENBQUMsdUJBQXVCO2FBQ25DO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLFVBQVUsRUFBRSxPQUFPLENBQUMsVUFBVSxJQUFJLFVBQVU7Z0JBQzVDLFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVyxJQUFJLHFCQUFxQjtnQkFDekQsT0FBTyxFQUFFLE9BQU87Z0JBQ2hCLEdBQUcsT0FBTyxDQUFDLG1CQUFtQjthQUMvQjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx5QkFBeUIsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQ3BELDBCQUFjLENBQUMsVUFBVSxDQUNVLENBQUM7UUFFdEMsNEZBQTRGO1FBQzVGLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUV2QyxNQUFNLG9CQUFvQixHQUFHLElBQUksQ0FBQyxRQUFRLENBQ3hDLElBQUksQ0FBQyxNQUFNLEVBQ1gsSUFBSSxDQUFDLHlCQUF5QixDQUFDLE1BQU0sQ0FDdEMsQ0FBQztRQUVGLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNsQiwrR0FBK0c7WUFDL0csNEdBQTRHO1lBQzVHLDBHQUEwRztZQUMxRyw2QkFBNkI7WUFDN0IsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsS0FBSywrQkFBa0IsQ0FBQyxJQUFJLEVBQUU7Z0JBQzNELElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxpQkFBUSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtvQkFDN0QsUUFBUSxFQUFFLElBQUk7b0JBQ2QsR0FBRyxFQUFFO3dCQUNILFFBQVEsRUFBRSxDQUFDLG9CQUFvQixDQUFDO3FCQUNqQztpQkFDRixDQUFDLENBQUM7YUFDSjtpQkFBTTtnQkFDTCxJQUFJLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxZQUFZLEVBQUU7b0JBQ2xDLFFBQVEsRUFBRSxDQUFDLG9CQUFvQixDQUFDO2lCQUNqQyxDQUFDLENBQUM7YUFDSjtZQUNELHFCQUFxQjtZQUNyQixJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDakUsK0dBQStHO1lBQy9HLDhHQUE4RztZQUM5RyxJQUFJLENBQUMscUJBQXFCLENBQUMsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLE9BQU8sQ0FBQyxDQUFDO1NBQzdEO2FBQU07WUFDTCwyREFBMkQ7WUFDM0QsSUFBSSxDQUFDLE9BQU8sQ0FDVixHQUFHLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxPQUFPLENBQUMsV0FBVyxXQUFXLG9CQUFvQixFQUFFLENBQ3ZGLENBQUM7WUFFRiwyRUFBMkU7WUFDM0UsSUFBSSxDQUFDLE1BQU0sRUFBRSxRQUFRLENBQUM7Z0JBQ3BCLHNCQUFzQixFQUFFO29CQUN0QixDQUFDO29CQUNELEVBQUUsTUFBTSxFQUFFLENBQUMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsRUFBRTtpQkFDakU7YUFDRixDQUFDLENBQUM7U0FDSjtRQUVELDJDQUEyQztRQUMzQyxNQUFNLGFBQWEsR0FBZ0M7WUFDakQseUJBQXlCLEVBQUUsZ0NBQWdDO1lBQzNELDJCQUEyQixFQUN6QixJQUFJLENBQUMseUJBQXlCLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDcEQsVUFBVSxFQUFFLE9BQU8sQ0FBQyxVQUFVO1lBQzlCLFNBQVMsRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUNqRCxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsa0JBQWtCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQjtTQUM1QyxDQUFDO1FBQ0YsSUFBSSxtQkFBVSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLEVBQUU7WUFDdkQsUUFBUSxFQUFFLG9CQUFvQixJQUFJLENBQUMsU0FBUyxJQUFJO1NBQ2pELENBQUMsQ0FBQztRQUNILElBQUksa0JBQVMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUMxRCxLQUFLLEVBQUUsc0NBQXlCLENBQUMsYUFBYSxDQUFDO1NBQ2hELENBQUMsQ0FBQztRQUNILElBQUksa0JBQVMsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNoQyxLQUFLLEVBQUUscUNBQXdCLENBQUMsYUFBYSxDQUFDO1NBQy9DLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNILGNBQWM7UUFDWiw4R0FBOEc7UUFDOUcsc0RBQXNEO1FBQ3RELCtFQUErRTtRQUMvRSxDQUFDLElBQUksQ0FBQyxTQUFTLElBQUksSUFBSSxDQUFDLCtCQUErQixFQUFFLENBQUM7UUFFMUQsS0FBSyxDQUFDLGNBQWMsRUFBRSxDQUFDO0lBQ3pCLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ssK0JBQStCO1FBQ3JDLFFBQVEsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUU7WUFDbkMsS0FBSywrQkFBa0IsQ0FBQyxHQUFHLENBQUM7WUFDNUIsS0FBSywrQkFBa0IsQ0FBQyxJQUFJO2dCQUMxQixXQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsT0FBTyxFQUFFO29CQUMxQyxHQUFHLEVBQUUsSUFBSSxDQUFDLHlCQUF5QixDQUFDLE1BQU07aUJBQzNDLENBQUMsQ0FBQztnQkFDSCxXQUFJLENBQ0YsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsU0FBUyxJQUFJLENBQUMseUJBQXlCLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxFQUMzRjtvQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLE1BQU07aUJBQ2pCLENBQ0YsQ0FBQztnQkFDRixNQUFNO1lBQ1IsS0FBSywrQkFBa0IsQ0FBQyxJQUFJO2dCQUMxQixXQUFJLENBQ0YsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLGNBQWMsV0FBVyxJQUFJLENBQUMsUUFBUSxDQUNwRCxJQUFJLENBQUMsTUFBTSxFQUNYLElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxNQUFNLENBQ3RDLEVBQUUsRUFDSDtvQkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLE1BQU07aUJBQ2pCLENBQ0YsQ0FBQztnQkFDRixNQUFNO1lBQ1I7Z0JBQ0UsTUFBTSxJQUFJLEtBQUssQ0FDYiwrQkFBK0IsSUFBSSxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsQ0FDN0QsQ0FBQztTQUNMO0lBQ0gsQ0FBQzs7QUE3T0gsMERBOE9DIiwic291cmNlc0NvbnRlbnQiOlsiLyoqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKlxuIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuXG4gTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS5cbiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlIHdpdGggdGhlIExpY2Vuc2UuXG4gWW91IG1heSBvYnRhaW4gYSBjb3B5IG9mIHRoZSBMaWNlbnNlIGF0XG5cbiBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcblxuIFVubGVzcyByZXF1aXJlZCBieSBhcHBsaWNhYmxlIGxhdyBvciBhZ3JlZWQgdG8gaW4gd3JpdGluZywgc29mdHdhcmVcbiBkaXN0cmlidXRlZCB1bmRlciB0aGUgTGljZW5zZSBpcyBkaXN0cmlidXRlZCBvbiBhbiBcIkFTIElTXCIgQkFTSVMsXG4gV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlciBleHByZXNzIG9yIGltcGxpZWQuXG4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zIGFuZFxuIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICoqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqICovXG5cbmltcG9ydCAqIGFzIHBhdGggZnJvbSBcInBhdGhcIjtcbmltcG9ydCB7IFByb2plY3QsIFNhbXBsZURpciwgU2FtcGxlRmlsZSwgVGV4dEZpbGUsIFlhbWxGaWxlIH0gZnJvbSBcInByb2plblwiO1xuaW1wb3J0IHsgTm9kZVBhY2thZ2VNYW5hZ2VyIH0gZnJvbSBcInByb2plbi9saWIvamF2YXNjcmlwdFwiO1xuaW1wb3J0IHtcbiAgVHlwZVNjcmlwdFByb2plY3QsXG4gIFR5cGVTY3JpcHRQcm9qZWN0T3B0aW9ucyxcbn0gZnJvbSBcInByb2plbi9saWIvdHlwZXNjcmlwdFwiO1xuaW1wb3J0IHsgZXhlYyB9IGZyb20gXCJwcm9qZW4vbGliL3V0aWxcIjtcbmltcG9ydCB7IGdlbmVyYXRlQ2xpZW50UHJvamVjdHMgfSBmcm9tIFwiLi9jb2RlZ2VuL2dlbmVyYXRlXCI7XG5pbXBvcnQgeyBHZW5lcmF0ZWRUeXBlc2NyaXB0Q2xpZW50UHJvamVjdCB9IGZyb20gXCIuL2NvZGVnZW4vZ2VuZXJhdGVkLXR5cGVzY3JpcHQtY2xpZW50LXByb2plY3RcIjtcbmltcG9ydCB7IENsaWVudExhbmd1YWdlIH0gZnJvbSBcIi4vbGFuZ3VhZ2VzXCI7XG5pbXBvcnQge1xuICBnZXRUeXBlc2NyaXB0U2FtcGxlU291cmNlLFxuICBnZXRUeXBlc2NyaXB0U2FtcGxlVGVzdHMsXG4gIFR5cGVzY3JpcHRTYW1wbGVDb2RlT3B0aW9ucyxcbn0gZnJvbSBcIi4vc2FtcGxlcy90eXBlc2NyaXB0XCI7XG5pbXBvcnQgeyBPcGVuQXBpU3BlY1Byb2plY3QgfSBmcm9tIFwiLi9zcGVjL29wZW4tYXBpLXNwZWMtcHJvamVjdFwiO1xuaW1wb3J0IHsgT3BlbkFwaUdhdGV3YXlQcm9qZWN0T3B0aW9ucyB9IGZyb20gXCIuL3R5cGVzXCI7XG5cbmNvbnN0IE9QRU5BUElfR0FURVdBWV9QREtfUEFDS0FHRV9OQU1FID1cbiAgXCJAYXdzLXByb3RvdHlwaW5nLXNkay9vcGVuLWFwaS1nYXRld2F5XCI7XG5cbi8qKlxuICogQ29uZmlndXJhdGlvbiBmb3IgdGhlIE9wZW5BcGlHYXRld2F5VHNQcm9qZWN0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT3BlbkFwaUdhdGV3YXlUc1Byb2plY3RPcHRpb25zXG4gIGV4dGVuZHMgVHlwZVNjcmlwdFByb2plY3RPcHRpb25zLFxuICAgIE9wZW5BcGlHYXRld2F5UHJvamVjdE9wdGlvbnMge31cblxuLyoqXG4gKiBTeW50aGVzaXplcyBhIFR5cGVzY3JpcHQgUHJvamVjdCB3aXRoIGFuIE9wZW5BUEkgc3BlYywgZ2VuZXJhdGVkIGNsaWVudHMsIGEgQ0RLIGNvbnN0cnVjdCBmb3IgZGVwbG95aW5nIHRoZSBBUElcbiAqIHdpdGggQVBJIEdhdGV3YXksIGFuZCBnZW5lcmF0ZWQgbGFtYmRhIGhhbmRsZXIgd3JhcHBlcnMgZm9yIHR5cGUtc2FmZSBoYW5kbGluZyBvZiByZXF1ZXN0cy5cbiAqXG4gKiBAcGppZCBvcGVuLWFwaS1nYXRld2F5LXRzXG4gKi9cbmV4cG9ydCBjbGFzcyBPcGVuQXBpR2F0ZXdheVRzUHJvamVjdCBleHRlbmRzIFR5cGVTY3JpcHRQcm9qZWN0IHtcbiAgLyoqXG4gICAqIEEgcmVmZXJlbmNlIHRvIHRoZSBnZW5lcmF0ZWQgdHlwZXNjcmlwdCBjbGllbnRcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBnZW5lcmF0ZWRUeXBlc2NyaXB0Q2xpZW50OiBUeXBlU2NyaXB0UHJvamVjdDtcblxuICAvKipcbiAgICogUmVmZXJlbmNlcyB0byB0aGUgY2xpZW50IHByb2plY3RzIHRoYXQgd2VyZSBnZW5lcmF0ZWQsIGtleWVkIGJ5IGxhbmd1YWdlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZ2VuZXJhdGVkQ2xpZW50czogeyBbbGFuZ3VhZ2U6IHN0cmluZ106IFByb2plY3QgfTtcblxuICAvKipcbiAgICogVGhlIGRpcmVjdG9yeSBpbiB3aGljaCB0aGUgT3BlbkFQSSBzcGVjIGZpbGUocykgcmVzaWRlLCByZWxhdGl2ZSB0byB0aGUgcHJvamVjdCBzcmNkaXJcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzcGVjRGlyOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkaXJlY3RvcnkgaW4gd2hpY2ggdGhlIGFwaSBnZW5lcmF0ZWQgY29kZSB3aWxsIHJlc2lkZSwgcmVsYXRpdmUgdG8gdGhlIHByb2plY3Qgc3JjZGlyXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYXBpU3JjRGlyOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBzcGVjIGZpbGVcbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBzcGVjRmlsZU5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRpcmVjdG9yeSBpbiB3aGljaCBnZW5lcmF0ZWQgY2xpZW50IGNvZGUgd2lsbCBiZSBnZW5lcmF0ZWQsIHJlbGF0aXZlIHRvIHRoZSBvdXRkaXIgb2YgdGhpcyBwcm9qZWN0XG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZ2VuZXJhdGVkQ29kZURpcjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSZWZlcmVuY2UgdG8gdGhlIFBOUE0gd29ya3NwYWNlIHlhbWwgZmlsZSB3aGljaCBhZGRzIHRoZSBkZXBlbmRlbmN5IGJldHdlZW4gdGhpcyBwcm9qZWN0IGFuZCB0aGUgZ2VuZXJhdGVkXG4gICAqIHR5cGVzY3JpcHQgY2xpZW50IHdoZW4gdGhpcyBwcm9qZWN0IGlzIHVzZWQgaW4gYSBtb25vcmVwbywgYW5kIHRoZSBwYWNrYWdlIG1hbmFnZXIgaXMgUE5QTS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBwbnBtV29ya3NwYWNlPzogWWFtbEZpbGU7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBoYXNQYXJlbnQ6IGJvb2xlYW47XG5cbiAgY29uc3RydWN0b3Iob3B0aW9uczogT3BlbkFwaUdhdGV3YXlUc1Byb2plY3RPcHRpb25zKSB7XG4gICAgc3VwZXIoe1xuICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIHNhbXBsZUNvZGU6IGZhbHNlLFxuICAgICAgdHNjb25maWc6IHtcbiAgICAgICAgY29tcGlsZXJPcHRpb25zOiB7XG4gICAgICAgICAgbGliOiBbXCJkb21cIiwgXCJlczIwMTlcIl0sXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgaWYgKG9wdGlvbnMuc3BlY0ZpbGUpIHtcbiAgICAgIHRoaXMuc3BlY0RpciA9IHBhdGguZGlybmFtZShvcHRpb25zLnNwZWNGaWxlKTtcbiAgICAgIHRoaXMuc3BlY0ZpbGVOYW1lID0gcGF0aC5iYXNlbmFtZShvcHRpb25zLnNwZWNGaWxlKTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5zcGVjRGlyID0gXCJzcGVjXCI7XG4gICAgICB0aGlzLnNwZWNGaWxlTmFtZSA9IFwic3BlYy55YW1sXCI7XG4gICAgfVxuICAgIHRoaXMuZ2VuZXJhdGVkQ29kZURpciA9IG9wdGlvbnMuZ2VuZXJhdGVkQ29kZURpciA/PyBcImdlbmVyYXRlZFwiO1xuICAgIHRoaXMuYXBpU3JjRGlyID0gb3B0aW9ucy5hcGlTcmNEaXIgPz8gXCJhcGlcIjtcblxuICAgIC8vIEFsbG93IGpzb24gZmlsZXMgdG8gYmUgaW1wb3J0ZWQgKGZvciBpbXBvcnRpbmcgdGhlIHBhcnNlZCBzcGVjKVxuICAgIHRoaXMudHNjb25maWc/LmFkZEluY2x1ZGUoYCR7dGhpcy5zcmNkaXJ9LyoqLyouanNvbmApO1xuXG4gICAgLy8gU2V0IHRvIHByaXZhdGUgc2luY2UgdGhpcyBlaXRoZXIgdXNlcyB3b3Jrc3BhY2VzIG9yIGhhcyBmaWxlIGRlcGVuZGVuY2llc1xuICAgIHRoaXMucGFja2FnZS5hZGRGaWVsZChcInByaXZhdGVcIiwgdHJ1ZSk7XG5cbiAgICAvLyBHZW5lcmF0ZWQgcHJvamVjdCBzaG91bGQgaGF2ZSBhIGRlcGVuZGVuY3kgb24gdGhpcyBwcm9qZWN0LCBpbiBvcmRlciB0byBydW4gdGhlIGdlbmVyYXRpb24gc2NyaXB0c1xuICAgIHRoaXMuYWRkRGVwcyhPUEVOQVBJX0dBVEVXQVlfUERLX1BBQ0tBR0VfTkFNRSwgXCJjb25zdHJ1Y3RzXCIsIFwiYXdzLWNkay1saWJcIik7XG5cbiAgICAvLyBTeW50aGVzaXplIHRoZSBvcGVuYXBpIHNwZWMgZWFybHkgc2luY2UgaXQncyB1c2VkIGJ5IHRoZSBnZW5lcmF0ZWQgdHlwZXNjcmlwdCBjbGllbnQsIHdoaWNoIGlzIGFsc28gc3ludGgnZCBlYXJseVxuICAgIGNvbnN0IHNwZWMgPSBuZXcgT3BlbkFwaVNwZWNQcm9qZWN0KHtcbiAgICAgIG5hbWU6IGAke3RoaXMubmFtZX0tc3BlY2AsXG4gICAgICBwYXJlbnQ6IHRoaXMsXG4gICAgICBvdXRkaXI6IHBhdGguam9pbih0aGlzLnNyY2RpciwgdGhpcy5zcGVjRGlyKSxcbiAgICAgIHNwZWNGaWxlTmFtZTogdGhpcy5zcGVjRmlsZU5hbWUsXG4gICAgICBwYXJzZWRTcGVjRmlsZU5hbWU6IG9wdGlvbnMucGFyc2VkU3BlY0ZpbGVOYW1lLFxuICAgIH0pO1xuICAgIHNwZWMuc3ludGgoKTtcblxuICAgIG5ldyBUZXh0RmlsZSh0aGlzLCBwYXRoLmpvaW4odGhpcy5nZW5lcmF0ZWRDb2RlRGlyLCBcIlJFQURNRS5tZFwiKSwge1xuICAgICAgbGluZXM6IFtcbiAgICAgICAgXCIjIyBHZW5lcmF0ZWQgQ2xpZW50c1wiLFxuICAgICAgICBcIlwiLFxuICAgICAgICBcIlRoaXMgZGlyZWN0b3J5IGNvbnRhaW5zIGdlbmVyYXRlZCBjbGllbnQgY29kZSBiYXNlZCBvbiB5b3VyIE9wZW5BUEkgU3BlY2lmaWNhdGlvbiBmaWxlIChzcGVjLnlhbWwpLlwiLFxuICAgICAgICBcIlwiLFxuICAgICAgICBcIkxpa2Ugb3RoZXIgYHByb2plbmAgbWFuYWdlZCBmaWxlcywgdGhpcyBkaXJlY3Rvcnkgc2hvdWxkIGJlIGNoZWNrZWQgaW4gdG8gc291cmNlIGNvbnRyb2wsIGJ1dCBzaG91bGQgbm90IGJlIGVkaXRlZCBtYW51YWxseS5cIixcbiAgICAgIF0sXG4gICAgICByZWFkb25seTogdHJ1ZSxcbiAgICB9KTtcblxuICAgIC8vIFBhcmVudCB0aGUgZ2VuZXJhdGVkIGNvZGUgd2l0aCB0aGlzIHByb2plY3QncyBwYXJlbnQgZm9yIGJldHRlciBpbnRlZ3JhdGlvbiB3aXRoIG1vbm9yZXBvc1xuICAgIHRoaXMuaGFzUGFyZW50ID0gISFvcHRpb25zLnBhcmVudDtcbiAgICBjb25zdCBnZW5lcmF0ZWRDb2RlRGlyUmVsYXRpdmVUb1BhcmVudCA9IHRoaXMuaGFzUGFyZW50XG4gICAgICA/IHBhdGguam9pbihvcHRpb25zLm91dGRpciEsIHRoaXMuZ2VuZXJhdGVkQ29kZURpcilcbiAgICAgIDogdGhpcy5nZW5lcmF0ZWRDb2RlRGlyO1xuXG4gICAgLy8gQWx3YXlzIGdlbmVyYXRlIHRoZSB0eXBlc2NyaXB0IGNsaWVudCBzaW5jZSB0aGlzIHByb2plY3Qgd2lsbCB0YWtlIGEgZGVwZW5kZW5jeSBvbiBpdCBpbiBvcmRlciB0byBwcm9kdWNlIHRoZVxuICAgIC8vIHR5cGUtc2FmZSBjZGsgY29uc3RydWN0IHdyYXBwZXIuXG4gICAgY29uc3QgY2xpZW50TGFuZ3VhZ2VzID0gbmV3IFNldChvcHRpb25zLmNsaWVudExhbmd1YWdlcyk7XG4gICAgY2xpZW50TGFuZ3VhZ2VzLmFkZChDbGllbnRMYW5ndWFnZS5UWVBFU0NSSVBUKTtcblxuICAgIHRoaXMuZ2VuZXJhdGVkQ2xpZW50cyA9IGdlbmVyYXRlQ2xpZW50UHJvamVjdHMoY2xpZW50TGFuZ3VhZ2VzLCB7XG4gICAgICBwYXJlbnQ6IHRoaXMuaGFzUGFyZW50ID8gb3B0aW9ucy5wYXJlbnQgOiB0aGlzLFxuICAgICAgcm9vdFByb2plY3RIYXNQYXJlbnQ6IHRoaXMuaGFzUGFyZW50LFxuICAgICAgcGFyZW50UGFja2FnZU5hbWU6IHRoaXMucGFja2FnZS5wYWNrYWdlTmFtZSxcbiAgICAgIGdlbmVyYXRlZENvZGVEaXI6IGdlbmVyYXRlZENvZGVEaXJSZWxhdGl2ZVRvUGFyZW50LFxuICAgICAgcGFyc2VkU3BlY1BhdGg6IHNwZWMucGFyc2VkU3BlY1BhdGgsXG4gICAgICB0eXBlc2NyaXB0T3B0aW9uczoge1xuICAgICAgICBkZWZhdWx0UmVsZWFzZUJyYW5jaDogb3B0aW9ucy5kZWZhdWx0UmVsZWFzZUJyYW5jaCxcbiAgICAgICAgcGFja2FnZU1hbmFnZXI6IG9wdGlvbnMucGFja2FnZU1hbmFnZXIsXG4gICAgICAgIC4uLm9wdGlvbnMudHlwZXNjcmlwdENsaWVudE9wdGlvbnMsXG4gICAgICB9LFxuICAgICAgcHl0aG9uT3B0aW9uczoge1xuICAgICAgICBhdXRob3JOYW1lOiBvcHRpb25zLmF1dGhvck5hbWUgPz8gXCJBUEogQ29wZVwiLFxuICAgICAgICBhdXRob3JFbWFpbDogb3B0aW9ucy5hdXRob3JFbWFpbCA/PyBcImFwai1jb3BlQGFtYXpvbi5jb21cIixcbiAgICAgICAgdmVyc2lvbjogXCIwLjAuMFwiLFxuICAgICAgICAuLi5vcHRpb25zLnB5dGhvbkNsaWVudE9wdGlvbnMsXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy5nZW5lcmF0ZWRUeXBlc2NyaXB0Q2xpZW50ID0gdGhpcy5nZW5lcmF0ZWRDbGllbnRzW1xuICAgICAgQ2xpZW50TGFuZ3VhZ2UuVFlQRVNDUklQVFxuICAgIF0gYXMgR2VuZXJhdGVkVHlwZXNjcmlwdENsaWVudFByb2plY3Q7XG5cbiAgICAvLyBTeW50aCBlYXJseSBzbyB0aGF0IHRoZSBnZW5lcmF0ZWQgY29kZSBpcyBhdmFpbGFibGUgcHJpb3IgdG8gdGhpcyBwcm9qZWN0J3MgaW5zdGFsbCBwaGFzZVxuICAgIHRoaXMuZ2VuZXJhdGVkVHlwZXNjcmlwdENsaWVudC5zeW50aCgpO1xuXG4gICAgY29uc3QgdHlwZXNjcmlwdENvZGVHZW5EaXIgPSBwYXRoLnJlbGF0aXZlKFxuICAgICAgdGhpcy5vdXRkaXIsXG4gICAgICB0aGlzLmdlbmVyYXRlZFR5cGVzY3JpcHRDbGllbnQub3V0ZGlyXG4gICAgKTtcblxuICAgIGlmICh0aGlzLmhhc1BhcmVudCkge1xuICAgICAgLy8gV2hlbiB3ZSBoYXZlIGEgcGFyZW50IHByb2plY3QsIHdlIGNhbiBhZGQgYSBkZXBlbmRlbmN5IG9uIHRoZSBnZW5lcmF0ZWQgdHlwZXNjcmlwdCBjbGllbnQgc2luY2UgaXQncyBwYXJ0IG9mXG4gICAgICAvLyB0aGUgbW9ub3JlcG8uIFNpbmNlIHRoaXMgcHJvamVjdCB3aWxsIGJlIHN5bnRoZXNpemVkIGJlZm9yZSB0aGUgcGFyZW50IG1vbm9yZXBvLCB0aGUgZ2VuZXJhdGVkIHR5cGVzY3JpcHRcbiAgICAgIC8vIGNsaWVudCB3b24ndCBiZSB2aXNpYmxlIGZvciB0aGUgZmlyc3QgaW5zdGFsbCBpbiB0aGlzIHByb2plY3QncyBwb3N0IHN5bnRoZXNpemUgc3RlcCwgc28gd2UgdXNlIGEgbG9jYWxcbiAgICAgIC8vIHdvcmtzcGFjZSBhcyBhIHdvcmthcm91bmQuXG4gICAgICBpZiAodGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyID09PSBOb2RlUGFja2FnZU1hbmFnZXIuUE5QTSkge1xuICAgICAgICB0aGlzLnBucG1Xb3Jrc3BhY2UgPSBuZXcgWWFtbEZpbGUodGhpcywgXCJwbnBtLXdvcmtzcGFjZS55YW1sXCIsIHtcbiAgICAgICAgICByZWFkb25seTogdHJ1ZSxcbiAgICAgICAgICBvYmo6IHtcbiAgICAgICAgICAgIHBhY2thZ2VzOiBbdHlwZXNjcmlwdENvZGVHZW5EaXJdLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhpcy5wYWNrYWdlLmFkZEZpZWxkKFwid29ya3NwYWNlc1wiLCB7XG4gICAgICAgICAgcGFja2FnZXM6IFt0eXBlc2NyaXB0Q29kZUdlbkRpcl0sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgICAgLy8gQWRkIHRoZSBkZXBlbmRlbmN5XG4gICAgICB0aGlzLmFkZERlcHModGhpcy5nZW5lcmF0ZWRUeXBlc2NyaXB0Q2xpZW50LnBhY2thZ2UucGFja2FnZU5hbWUpO1xuICAgICAgLy8gU2luY2UgdGhlIGdlbmVyYXRlZCB0eXBlc2NyaXB0IGNsaWVudCBwcm9qZWN0IGlzIHBhcmVudGVkIGJ5IHRoaXMgcHJvamVjdCdzIHBhcmVudCByYXRoZXIgdGhhbiB0aGlzIHByb2plY3QsXG4gICAgICAvLyBwcm9qZW4gd2lsbCBjbGVhbiB1cCB0aGUgZ2VuZXJhdGVkIGNsaWVudCB3aGVuIHN5bnRoZXNpemluZyB0aGlzIHByb2plY3QgdW5sZXNzIHdlIGFkZCBhbiBleHBsaWNpdCBleGNsdWRlLlxuICAgICAgdGhpcy5hZGRFeGNsdWRlRnJvbUNsZWFudXAoYCR7dGhpcy5nZW5lcmF0ZWRDb2RlRGlyfS8qKi8qYCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIEFkZCBhIGZpbGUgZGVwZW5kZW5jeSBvbiB0aGUgZ2VuZXJhdGVkIHR5cGVzY3JpcHQgY2xpZW50XG4gICAgICB0aGlzLmFkZERlcHMoXG4gICAgICAgIGAke3RoaXMuZ2VuZXJhdGVkVHlwZXNjcmlwdENsaWVudC5wYWNrYWdlLnBhY2thZ2VOYW1lfUBmaWxlOi4vJHt0eXBlc2NyaXB0Q29kZUdlbkRpcn1gXG4gICAgICApO1xuXG4gICAgICAvLyBMaW50aW5nIHJ1bnMgcG9zdCBzeW50aGVzaXplIGJlZm9yZSB0aGUgdHlwZXNjcmlwdCBjbGllbnQgaGFzIGJlZW4gYnVpbHRcbiAgICAgIHRoaXMuZXNsaW50Py5hZGRSdWxlcyh7XG4gICAgICAgIFwiaW1wb3J0L25vLXVucmVzb2x2ZWRcIjogW1xuICAgICAgICAgIDIsXG4gICAgICAgICAgeyBpZ25vcmU6IFt0aGlzLmdlbmVyYXRlZFR5cGVzY3JpcHRDbGllbnQucGFja2FnZS5wYWNrYWdlTmFtZV0gfSxcbiAgICAgICAgXSxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIC8vIEdlbmVyYXRlIHRoZSBzYW1wbGUgc291cmNlIGFuZCB0ZXN0IGNvZGVcbiAgICBjb25zdCBzYW1wbGVPcHRpb25zOiBUeXBlc2NyaXB0U2FtcGxlQ29kZU9wdGlvbnMgPSB7XG4gICAgICBvcGVuQXBpR2F0ZXdheVBhY2thZ2VOYW1lOiBPUEVOQVBJX0dBVEVXQVlfUERLX1BBQ0tBR0VfTkFNRSxcbiAgICAgIHR5cGVzY3JpcHRDbGllbnRQYWNrYWdlTmFtZTpcbiAgICAgICAgdGhpcy5nZW5lcmF0ZWRUeXBlc2NyaXB0Q2xpZW50LnBhY2thZ2UucGFja2FnZU5hbWUsXG4gICAgICBzYW1wbGVDb2RlOiBvcHRpb25zLnNhbXBsZUNvZGUsXG4gICAgICBhcGlTcmNEaXI6IHBhdGguam9pbih0aGlzLnNyY2RpciwgdGhpcy5hcGlTcmNEaXIpLFxuICAgICAgc3BlY0RpcjogdGhpcy5zcGVjRGlyLFxuICAgICAgcGFyc2VkU3BlY0ZpbGVOYW1lOiBzcGVjLnBhcnNlZFNwZWNGaWxlTmFtZSxcbiAgICB9O1xuICAgIG5ldyBTYW1wbGVGaWxlKHRoaXMsIHBhdGguam9pbih0aGlzLnNyY2RpciwgXCJpbmRleC50c1wiKSwge1xuICAgICAgY29udGVudHM6IGBleHBvcnQgKiBmcm9tIFwiLi8ke3RoaXMuYXBpU3JjRGlyfVwiO2AsXG4gICAgfSk7XG4gICAgbmV3IFNhbXBsZURpcih0aGlzLCBwYXRoLmpvaW4odGhpcy5zcmNkaXIsIHRoaXMuYXBpU3JjRGlyKSwge1xuICAgICAgZmlsZXM6IGdldFR5cGVzY3JpcHRTYW1wbGVTb3VyY2Uoc2FtcGxlT3B0aW9ucyksXG4gICAgfSk7XG4gICAgbmV3IFNhbXBsZURpcih0aGlzLCB0aGlzLnRlc3RkaXIsIHtcbiAgICAgIGZpbGVzOiBnZXRUeXBlc2NyaXB0U2FtcGxlVGVzdHMoc2FtcGxlT3B0aW9ucyksXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogQGluaGVyaXREb2NcbiAgICovXG4gIHBvc3RTeW50aGVzaXplKCkge1xuICAgIC8vIFdoZW4gbm8gcGFyZW50IGlzIHBhc3NlZCwgbGluayB0aGUgZ2VuZXJhdGVkIGNsaWVudCBhcyBhIHByZWJ1aWxkIHN0ZXAgdG8gZW5zdXJlIHRoZSBsYXRlc3QgYnVpbHQgZ2VuZXJhdGVkXG4gICAgLy8gY2xpZW50IGlzIHJlZmxlY3RlZCBpbiB0aGlzIHBhY2thZ2UncyBub2RlIG1vZHVsZXMuXG4gICAgLy8gTm90ZSB0aGF0IGl0J3MgdXAgdG8gdGhlIHVzZXIgdG8gbWFuYWdlIGJ1aWxkaW5nIHRoZSBnZW5lcmF0ZWQgY2xpZW50IGZpcnN0LlxuICAgICF0aGlzLmhhc1BhcmVudCAmJiB0aGlzLmV4ZWN1dGVMaW5rTmF0aXZlQ2xpZW50Q29tbWFuZHMoKTtcblxuICAgIHN1cGVyLnBvc3RTeW50aGVzaXplKCk7XG4gIH1cblxuICAvKipcbiAgICogRXhlY3V0ZXMgY29tbWFuZHMgdG8gbGluayB0aGUgbmF0aXZlIChpZSB0eXBlc2NyaXB0KSBjbGllbnQgc3VjaCB0aGF0IHVwZGF0ZXMgdG8gdGhlIGdlbmVyYXRlZCBjbGllbnQgYXJlIHJlZmxlY3RlZFxuICAgKiBpbiB0aGlzIHByb2plY3QncyBub2RlX21vZHVsZXMgKHdoZW4gbm90IG1hbmFnZWQgYnkgYSBtb25vcmVwbylcbiAgICogQHByaXZhdGVcbiAgICovXG4gIHByaXZhdGUgZXhlY3V0ZUxpbmtOYXRpdmVDbGllbnRDb21tYW5kcygpIHtcbiAgICBzd2l0Y2ggKHRoaXMucGFja2FnZS5wYWNrYWdlTWFuYWdlcikge1xuICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuTlBNOlxuICAgICAgY2FzZSBOb2RlUGFja2FnZU1hbmFnZXIuWUFSTjpcbiAgICAgICAgZXhlYyhgJHt0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXJ9IGxpbmtgLCB7XG4gICAgICAgICAgY3dkOiB0aGlzLmdlbmVyYXRlZFR5cGVzY3JpcHRDbGllbnQub3V0ZGlyLFxuICAgICAgICB9KTtcbiAgICAgICAgZXhlYyhcbiAgICAgICAgICBgJHt0aGlzLnBhY2thZ2UucGFja2FnZU1hbmFnZXJ9IGxpbmsgJHt0aGlzLmdlbmVyYXRlZFR5cGVzY3JpcHRDbGllbnQucGFja2FnZS5wYWNrYWdlTmFtZX1gLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGN3ZDogdGhpcy5vdXRkaXIsXG4gICAgICAgICAgfVxuICAgICAgICApO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgTm9kZVBhY2thZ2VNYW5hZ2VyLlBOUE06XG4gICAgICAgIGV4ZWMoXG4gICAgICAgICAgYCR7dGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyfSBsaW5rIC4vJHtwYXRoLnJlbGF0aXZlKFxuICAgICAgICAgICAgdGhpcy5vdXRkaXIsXG4gICAgICAgICAgICB0aGlzLmdlbmVyYXRlZFR5cGVzY3JpcHRDbGllbnQub3V0ZGlyXG4gICAgICAgICAgKX1gLFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIGN3ZDogdGhpcy5vdXRkaXIsXG4gICAgICAgICAgfVxuICAgICAgICApO1xuICAgICAgICBicmVhaztcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcbiAgICAgICAgICBgVW5zdXBwb3J0ZWQgcGFja2FnZSBtYW5hZ2VyICR7dGhpcy5wYWNrYWdlLnBhY2thZ2VNYW5hZ2VyfWBcbiAgICAgICAgKTtcbiAgICB9XG4gIH1cbn1cbiJdfQ==