"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Integrations = exports.LambdaIntegration = exports.Integration = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const utils_1 = require("../spec/utils");
/**
 * An integration for an API operation. You can extend this to implement your own integration if you like.
 */
class Integration {
    /**
     * Grant permissions for the API to invoke the integration
     */
    grant(_props) {
        // No permissions to grant by default
    }
}
exports.Integration = Integration;
_a = JSII_RTTI_SYMBOL_1;
Integration[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.Integration", version: "0.11.2" };
/**
 * A lambda integration
 */
class LambdaIntegration extends Integration {
    constructor(lambdaFunction) {
        super();
        this.lambdaFunction = lambdaFunction;
    }
    /**
     * Render the lambda integration as a snippet of OpenAPI
     */
    render(props) {
        return {
            type: "AWS_PROXY",
            httpMethod: "POST",
            uri: utils_1.functionInvocationUri(props.scope, this.lambdaFunction),
            passthroughBehavior: "WHEN_NO_MATCH",
        };
    }
    /**
     * Grant API Gateway permissions to invoke the lambda
     */
    grant({ scope, api, operationId, method, path, }) {
        new aws_lambda_1.CfnPermission(scope, `LambdaPermission-${operationId}`, {
            action: "lambda:InvokeFunction",
            principal: "apigateway.amazonaws.com",
            functionName: this.lambdaFunction.functionArn,
            sourceArn: aws_cdk_lib_1.Stack.of(scope).formatArn({
                service: "execute-api",
                resource: api.restApiId,
                // Scope permissions to the stage, method and path of the operation.
                // Path parameters (eg {param} are replaced with wildcards)
                resourceName: `${api.deploymentStage.stageName}/${method.toUpperCase()}${path.replace(/{[^\}]*\}/g, "*")}`,
            }),
        });
    }
}
exports.LambdaIntegration = LambdaIntegration;
_b = JSII_RTTI_SYMBOL_1;
LambdaIntegration[_b] = { fqn: "@aws-prototyping-sdk/open-api-gateway.LambdaIntegration", version: "0.11.2" };
/**
 * A collection of integrations to connect API operations with a backend to service requests
 */
class Integrations {
    /**
     * An integration that invokes a lambda function to service the request
     * @param lambdaFunction the function to invoke
     */
    static lambda(lambdaFunction) {
        return new LambdaIntegration(lambdaFunction);
    }
}
exports.Integrations = Integrations;
_c = JSII_RTTI_SYMBOL_1;
Integrations[_c] = { fqn: "@aws-prototyping-sdk/open-api-gateway.Integrations", version: "0.11.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW50ZWdyYXRpb25zLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2NvbnN0cnVjdC9pbnRlZ3JhdGlvbnMvaW50ZWdyYXRpb25zLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUE7Ozs7Ozs7Ozs7Ozs7O3dIQWN3SDtBQUN4SCw2Q0FBb0M7QUFFcEMsdURBQWtFO0FBR2xFLHlDQUFzRDtBQXFNdEQ7O0dBRUc7QUFDSCxNQUFzQixXQUFXO0lBSy9COztPQUVHO0lBQ0ksS0FBSyxDQUFDLE1BQTZCO1FBQ3hDLHFDQUFxQztJQUN2QyxDQUFDOztBQVZILGtDQVdDOzs7QUFFRDs7R0FFRztBQUNILE1BQWEsaUJBQWtCLFNBQVEsV0FBVztJQUdoRCxZQUFZLGNBQXlCO1FBQ25DLEtBQUssRUFBRSxDQUFDO1FBQ1IsSUFBSSxDQUFDLGNBQWMsR0FBRyxjQUFjLENBQUM7SUFDdkMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEtBQTZCO1FBQ3pDLE9BQU87WUFDTCxJQUFJLEVBQUUsV0FBVztZQUNqQixVQUFVLEVBQUUsTUFBTTtZQUNsQixHQUFHLEVBQUUsNkJBQXFCLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxJQUFJLENBQUMsY0FBYyxDQUFDO1lBQzVELG1CQUFtQixFQUFFLGVBQWU7U0FDckMsQ0FBQztJQUNKLENBQUM7SUFFRDs7T0FFRztJQUNJLEtBQUssQ0FBQyxFQUNYLEtBQUssRUFDTCxHQUFHLEVBQ0gsV0FBVyxFQUNYLE1BQU0sRUFDTixJQUFJLEdBQ2tCO1FBQ3RCLElBQUksMEJBQWEsQ0FBQyxLQUFLLEVBQUUsb0JBQW9CLFdBQVcsRUFBRSxFQUFFO1lBQzFELE1BQU0sRUFBRSx1QkFBdUI7WUFDL0IsU0FBUyxFQUFFLDBCQUEwQjtZQUNyQyxZQUFZLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxXQUFXO1lBQzdDLFNBQVMsRUFBRSxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLENBQUM7Z0JBQ25DLE9BQU8sRUFBRSxhQUFhO2dCQUN0QixRQUFRLEVBQUUsR0FBRyxDQUFDLFNBQVM7Z0JBQ3ZCLG9FQUFvRTtnQkFDcEUsMkRBQTJEO2dCQUMzRCxZQUFZLEVBQUUsR0FDWixHQUFHLENBQUMsZUFBZSxDQUFDLFNBQ3RCLElBQUksTUFBTSxDQUFDLFdBQVcsRUFBRSxHQUFHLElBQUksQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLEdBQUcsQ0FBQyxFQUFFO2FBQzdELENBQUM7U0FDSCxDQUFDLENBQUM7SUFDTCxDQUFDOztBQTVDSCw4Q0E2Q0M7OztBQUVEOztHQUVHO0FBQ0gsTUFBYSxZQUFZO0lBQ3ZCOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxNQUFNLENBQUMsY0FBeUI7UUFDNUMsT0FBTyxJQUFJLGlCQUFpQixDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQy9DLENBQUM7O0FBUEgsb0NBUUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqXG4gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5cbiBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLlxuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2Ugd2l0aCB0aGUgTGljZW5zZS5cbiBZb3UgbWF5IG9idGFpbiBhIGNvcHkgb2YgdGhlIExpY2Vuc2UgYXRcblxuIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuXG4gVW5sZXNzIHJlcXVpcmVkIGJ5IGFwcGxpY2FibGUgbGF3IG9yIGFncmVlZCB0byBpbiB3cml0aW5nLCBzb2Z0d2FyZVxuIGRpc3RyaWJ1dGVkIHVuZGVyIHRoZSBMaWNlbnNlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuIFwiQVMgSVNcIiBCQVNJUyxcbiBXSVRIT1VUIFdBUlJBTlRJRVMgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZWl0aGVyIGV4cHJlc3Mgb3IgaW1wbGllZC5cbiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnMgYW5kXG4gbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKiogKi9cbmltcG9ydCB7IFN0YWNrIH0gZnJvbSBcImF3cy1jZGstbGliXCI7XG5pbXBvcnQgeyBTcGVjUmVzdEFwaSB9IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtYXBpZ2F0ZXdheVwiO1xuaW1wb3J0IHsgQ2ZuUGVybWlzc2lvbiwgSUZ1bmN0aW9uIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1sYW1iZGFcIjtcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgeyBNZXRob2RBbmRQYXRoIH0gZnJvbSBcIi4uL3NwZWNcIjtcbmltcG9ydCB7IGZ1bmN0aW9uSW52b2NhdGlvblVyaSB9IGZyb20gXCIuLi9zcGVjL3V0aWxzXCI7XG5cbi8qKlxuICogU3BlY2lmaWVzIGFuIEFQSSBtZXRob2QgaW50ZWdyYXRpb24gdHlwZVxuICovXG5leHBvcnQgdHlwZSBBcGlHYXRld2F5SW50ZWdyYXRpb25UeXBlID1cbiAgfCBcIkhUVFBcIlxuICB8IFwiSFRUUF9QUk9YWVwiXG4gIHwgXCJBV1NfUFJPWFlcIlxuICB8IFwiQVdTXCJcbiAgfCBcIk1PQ0tcIjtcblxuLyoqXG4gKiBUaGUgdHlwZSBvZiB0aGUgbmV0d29yayBjb25uZWN0aW9uIHRvIHRoZSBpbnRlZ3JhdGlvbiBlbmRwb2ludC4gVGhlIHZhbGlkIHZhbHVlIGlzIElOVEVSTkVUIGZvciBjb25uZWN0aW9ucyB0aHJvdWdoXG4gKiB0aGUgcHVibGljIHJvdXRhYmxlIGludGVybmV0IG9yIFZQQ19MSU5LIGZvciBwcml2YXRlIGNvbm5lY3Rpb25zIGJldHdlZW4gQVBJIEdhdGV3YXkgYW5kIGEgbmV0d29yayBsb2FkIGJhbGFuY2VyIGluXG4gKiBhIFZQQy4gVGhlIGRlZmF1bHQgdmFsdWUgaXMgSU5URVJORVQuXG4gKi9cbmV4cG9ydCB0eXBlIEFwaUdhdGV3YXlJbnRlZ3JhdGlvbkNvbm5lY3Rpb25UeXBlID0gXCJWUENfTElOS1wiIHwgXCJJTlRFUk5FVFwiO1xuXG4vKipcbiAqIFNwZWNpZmllcyBob3cgdG8gaGFuZGxlIHJlcXVlc3QgcGF5bG9hZCBjb250ZW50IHR5cGUgY29udmVyc2lvbnMuXG4gKi9cbmV4cG9ydCB0eXBlIEFwaUdhdGV3YXlJbnRlZ3JhdGlvbkNvbnRlbnRIYW5kbGluZyA9XG4gIHwgXCJDT05WRVJUX1RPX1RFWFRcIlxuICB8IFwiQ09OVkVSVF9UT19CSU5BUllcIjtcblxuLyoqXG4gKiBTcGVjaWZpZXMgaG93IHRoZSBtZXRob2QgcmVxdWVzdCBib2R5IG9mIGFuIHVubWFwcGVkIGNvbnRlbnQgdHlwZSB3aWxsIGJlIHBhc3NlZCB0aHJvdWdoIHRoZSBpbnRlZ3JhdGlvbiByZXF1ZXN0IHRvXG4gKiB0aGUgYmFjayBlbmQgd2l0aG91dCB0cmFuc2Zvcm1hdGlvbi5cbiAqL1xuZXhwb3J0IHR5cGUgQXBpR2F0ZXdheUludGVncmF0aW9uUGFzc3Rocm91Z2hCZWhhdmlvdXIgPVxuICB8IFwiV0hFTl9OT19NQVRDSFwiXG4gIHwgXCJXSEVOX05PX1RFTVBMQVRFU1wiXG4gIHwgXCJORVZFUlwiO1xuXG4vKipcbiAqIEFQSSBHYXRld2F5IGludGVncmF0aW9uIHJlc3BvbnNlXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1zd2FnZ2VyLWV4dGVuc2lvbnMtaW50ZWdyYXRpb24tcmVzcG9uc2UuaHRtbFxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9JbnRlZ3JhdGlvbi5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBpR2F0ZXdheUludGVncmF0aW9uUmVzcG9uc2Uge1xuICAvKipcbiAgICogSFRUUCBzdGF0dXMgY29kZSBmb3IgdGhlIG1ldGhvZCByZXNwb25zZVxuICAgKi9cbiAgcmVhZG9ubHkgc3RhdHVzQ29kZTogc3RyaW5nO1xuICAvKipcbiAgICogU3BlY2lmaWVzIE1JTUUgdHlwZS1zcGVjaWZpYyBtYXBwaW5nIHRlbXBsYXRlcyBmb3IgdGhlIHJlc3BvbnNl4oCZcyBwYXlsb2FkLlxuICAgKi9cbiAgcmVhZG9ubHkgcmVzcG9uc2VUZW1wbGF0ZXM6IHsgW21pbWVUeXBlOiBzdHJpbmddOiBzdHJpbmcgfTtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyBwYXJhbWV0ZXIgbWFwcGluZ3MgZm9yIHRoZSByZXNwb25zZS5cbiAgICovXG4gIHJlYWRvbmx5IHJlc3BvbnNlUGFyYW1ldGVyczogeyBbcHJvcGVydHk6IHN0cmluZ106IHN0cmluZyB9O1xuICAvKipcbiAgICogUmVzcG9uc2UgcGF5bG9hZCBlbmNvZGluZyBjb252ZXJzaW9uIHR5cGVzLiBWYWxpZCB2YWx1ZXMgYXJlIDEpIENPTlZFUlRfVE9fVEVYVCwgZm9yIGNvbnZlcnRpbmcgYSBiaW5hcnkgcGF5bG9hZFxuICAgKiBpbnRvIGEgYmFzZTY0LWVuY29kZWQgc3RyaW5nIG9yIGNvbnZlcnRpbmcgYSB0ZXh0IHBheWxvYWQgaW50byBhIHV0Zi04LWVuY29kZWQgc3RyaW5nIG9yIHBhc3NpbmcgdGhyb3VnaCB0aGUgdGV4dFxuICAgKiBwYXlsb2FkIG5hdGl2ZWx5IHdpdGhvdXQgbW9kaWZpY2F0aW9uLCBhbmQgMikgQ09OVkVSVF9UT19CSU5BUlksIGZvciBjb252ZXJ0aW5nIGEgdGV4dCBwYXlsb2FkIGludG8gYVxuICAgKiBiYXNlNjQtZGVjb2RlZCBibG9iIG9yIHBhc3NpbmcgdGhyb3VnaCBhIGJpbmFyeSBwYXlsb2FkIG5hdGl2ZWx5IHdpdGhvdXQgbW9kaWZpY2F0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgY29udGVudEhhbmRsaW5nPzogQXBpR2F0ZXdheUludGVncmF0aW9uQ29udGVudEhhbmRsaW5nO1xufVxuXG4vKipcbiAqIFNwZWNpZmllcyB0aGUgVExTIGNvbmZpZ3VyYXRpb24gZm9yIGFuIGludGVncmF0aW9uXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1leHRlbnNpb25zLWludGVncmF0aW9uLXRscy1jb25maWcuaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEFwaUdhdGV3YXlJbnRlZ3JhdGlvblRsc0NvbmZpZyB7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgd2hldGhlciBvciBub3QgQVBJIEdhdGV3YXkgc2tpcHMgdmVyaWZpY2F0aW9uIHRoYXQgdGhlIGNlcnRpZmljYXRlIGZvciBhbiBpbnRlZ3JhdGlvbiBlbmRwb2ludCBpcyBpc3N1ZWRcbiAgICogYnkgYSBzdXBwb3J0ZWQgY2VydGlmaWNhdGUgYXV0aG9yaXR5LiBUaGlzIGlzbuKAmXQgcmVjb21tZW5kZWQsIGJ1dCBpdCBlbmFibGVzIHlvdSB0byB1c2UgY2VydGlmaWNhdGVzIHRoYXQgYXJlXG4gICAqIHNpZ25lZCBieSBwcml2YXRlIGNlcnRpZmljYXRlIGF1dGhvcml0aWVzLCBvciBjZXJ0aWZpY2F0ZXMgdGhhdCBhcmUgc2VsZi1zaWduZWQuIElmIGVuYWJsZWQsIEFQSSBHYXRld2F5IHN0aWxsXG4gICAqIHBlcmZvcm1zIGJhc2ljIGNlcnRpZmljYXRlIHZhbGlkYXRpb24sIHdoaWNoIGluY2x1ZGVzIGNoZWNraW5nIHRoZSBjZXJ0aWZpY2F0ZSdzIGV4cGlyYXRpb24gZGF0ZSwgaG9zdG5hbWUsIGFuZFxuICAgKiBwcmVzZW5jZSBvZiBhIHJvb3QgY2VydGlmaWNhdGUgYXV0aG9yaXR5LiBTdXBwb3J0ZWQgb25seSBmb3IgSFRUUCBhbmQgSFRUUF9QUk9YWSBpbnRlZ3JhdGlvbnMuXG4gICAqL1xuICByZWFkb25seSBpbnNlY3VyZVNraXBWZXJpZmljYXRpb24/OiBib29sZWFuO1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYW4gYXBpIGdhdGV3YXkgaW50ZWdyYXRpb25cbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwaWdhdGV3YXkvbGF0ZXN0L2RldmVsb3Blcmd1aWRlL2FwaS1nYXRld2F5LXN3YWdnZXItZXh0ZW5zaW9ucy1pbnRlZ3JhdGlvbi5odG1sXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBpR2F0ZXdheUludGVncmF0aW9uIHtcbiAgLyoqXG4gICAqIFRoZSB0eXBlIG9mIGludGVncmF0aW9uIHdpdGggdGhlIHNwZWNpZmllZCBiYWNrZW5kLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX0ludGVncmF0aW9uLmh0bWwjdHlwZVxuICAgKi9cbiAgcmVhZG9ubHkgdHlwZT86IEFwaUdhdGV3YXlJbnRlZ3JhdGlvblR5cGU7XG4gIC8qKlxuICAgKiBUaGUgZW5kcG9pbnQgVVJJIG9mIHRoZSBiYWNrZW5kLiBGb3IgaW50ZWdyYXRpb25zIG9mIHRoZSBhd3MgdHlwZSwgdGhpcyBpcyBhbiBBUk4gdmFsdWUuXG4gICAqIEZvciB0aGUgSFRUUCBpbnRlZ3JhdGlvbiwgdGhpcyBpcyB0aGUgVVJMIG9mIHRoZSBIVFRQIGVuZHBvaW50IGluY2x1ZGluZyB0aGUgaHR0cHMgb3IgaHR0cCBzY2hlbWUuXG4gICAqL1xuICByZWFkb25seSB1cmk/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgcmVxdWVzdCBwYXJhbWV0ZXJzIHdob3NlIHZhbHVlcyBhcmUgdG8gYmUgY2FjaGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgY2FjaGVLZXlQYXJhbWV0ZXJzPzogc3RyaW5nW107XG4gIC8qKlxuICAgKiBBbiBBUEktc3BlY2lmaWMgdGFnIGdyb3VwIG9mIHJlbGF0ZWQgY2FjaGVkIHBhcmFtZXRlcnMuXG4gICAqL1xuICByZWFkb25seSBjYWNoZU5hbWVzcGFjZT86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBJRCBvZiBhIFZwY0xpbmsgZm9yIHRoZSBwcml2YXRlIGludGVncmF0aW9uLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9hcGkvQVBJX1ZwY0xpbmsuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgY29ubmVjdGlvbklkPzogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIGludGVncmF0aW9uIGNvbm5lY3Rpb24gdHlwZS4gVGhlIHZhbGlkIHZhbHVlIGlzIFwiVlBDX0xJTktcIiBmb3IgcHJpdmF0ZSBpbnRlZ3JhdGlvbiBvciBcIklOVEVSTkVUXCIsIG90aGVyd2lzZS5cbiAgICovXG4gIHJlYWRvbmx5IGNvbm5lY3Rpb25UeXBlPzogQXBpR2F0ZXdheUludGVncmF0aW9uQ29ubmVjdGlvblR5cGU7XG4gIC8qKlxuICAgKiBGb3IgQVdTIElBTSByb2xlLWJhc2VkIGNyZWRlbnRpYWxzLCBzcGVjaWZ5IHRoZSBBUk4gb2YgYW4gYXBwcm9wcmlhdGUgSUFNIHJvbGUuIElmIHVuc3BlY2lmaWVkLCBjcmVkZW50aWFscyBkZWZhdWx0XG4gICAqIHRvIHJlc291cmNlLWJhc2VkIHBlcm1pc3Npb25zIHRoYXQgbXVzdCBiZSBhZGRlZCBtYW51YWxseSB0byBhbGxvdyB0aGUgQVBJIHRvIGFjY2VzcyB0aGUgcmVzb3VyY2UuIEZvciBtb3JlXG4gICAqIGluZm9ybWF0aW9uLCBzZWUgR3JhbnRpbmcgUGVybWlzc2lvbnMgVXNpbmcgYSBSZXNvdXJjZSBQb2xpY3kuXG4gICAqXG4gICAqIE5vdGU6IFdoZW4gdXNpbmcgSUFNIGNyZWRlbnRpYWxzLCBtYWtlIHN1cmUgdGhhdCBBV1MgU1RTIFJlZ2lvbmFsIGVuZHBvaW50cyBhcmUgZW5hYmxlZCBmb3IgdGhlIFJlZ2lvbiB3aGVyZSB0aGlzXG4gICAqIEFQSSBpcyBkZXBsb3llZCBmb3IgYmVzdCBwZXJmb3JtYW5jZS5cbiAgICovXG4gIHJlYWRvbmx5IGNyZWRlbnRpYWxzPzogc3RyaW5nO1xuICAvKipcbiAgICogUmVxdWVzdCBwYXlsb2FkIGVuY29kaW5nIGNvbnZlcnNpb24gdHlwZXMuIFZhbGlkIHZhbHVlcyBhcmUgMSkgQ09OVkVSVF9UT19URVhULCBmb3IgY29udmVydGluZyBhIGJpbmFyeSBwYXlsb2FkXG4gICAqIGludG8gYSBiYXNlNjQtZW5jb2RlZCBzdHJpbmcgb3IgY29udmVydGluZyBhIHRleHQgcGF5bG9hZCBpbnRvIGEgdXRmLTgtZW5jb2RlZCBzdHJpbmcgb3IgcGFzc2luZyB0aHJvdWdoIHRoZSB0ZXh0XG4gICAqIHBheWxvYWQgbmF0aXZlbHkgd2l0aG91dCBtb2RpZmljYXRpb24sIGFuZCAyKSBDT05WRVJUX1RPX0JJTkFSWSwgZm9yIGNvbnZlcnRpbmcgYSB0ZXh0IHBheWxvYWQgaW50byBhXG4gICAqIGJhc2U2NC1kZWNvZGVkIGJsb2Igb3IgcGFzc2luZyB0aHJvdWdoIGEgYmluYXJ5IHBheWxvYWQgbmF0aXZlbHkgd2l0aG91dCBtb2RpZmljYXRpb24uXG4gICAqL1xuICByZWFkb25seSBjb250ZW50SGFuZGxpbmc/OiBBcGlHYXRld2F5SW50ZWdyYXRpb25Db250ZW50SGFuZGxpbmc7XG4gIC8qKlxuICAgKiBUaGUgSFRUUCBtZXRob2QgdXNlZCBpbiB0aGUgaW50ZWdyYXRpb24gcmVxdWVzdC4gRm9yIExhbWJkYSBmdW5jdGlvbiBpbnZvY2F0aW9ucywgdGhlIHZhbHVlIG11c3QgYmUgUE9TVC5cbiAgICovXG4gIHJlYWRvbmx5IGh0dHBNZXRob2Q/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgaG93IGEgcmVxdWVzdCBwYXlsb2FkIG9mIHVubWFwcGVkIGNvbnRlbnQgdHlwZSBpcyBwYXNzZWQgdGhyb3VnaCB0aGUgaW50ZWdyYXRpb24gcmVxdWVzdCB3aXRob3V0XG4gICAqIG1vZGlmaWNhdGlvbi4gU3VwcG9ydGVkIHZhbHVlcyBhcmUgd2hlbl9ub190ZW1wbGF0ZXMsIHdoZW5fbm9fbWF0Y2gsIGFuZCBuZXZlci5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvYXBpL0FQSV9JbnRlZ3JhdGlvbi5odG1sI3Bhc3N0aHJvdWdoQmVoYXZpb3JcbiAgICovXG4gIHJlYWRvbmx5IHBhc3N0aHJvdWdoQmVoYXZpb3I/OiBBcGlHYXRld2F5SW50ZWdyYXRpb25QYXNzdGhyb3VnaEJlaGF2aW91cjtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyBtYXBwaW5ncyBmcm9tIG1ldGhvZCByZXF1ZXN0IHBhcmFtZXRlcnMgdG8gaW50ZWdyYXRpb24gcmVxdWVzdCBwYXJhbWV0ZXJzLiBTdXBwb3J0ZWQgcmVxdWVzdCBwYXJhbWV0ZXJzXG4gICAqIGFyZSBxdWVyeXN0cmluZywgcGF0aCwgaGVhZGVyLCBhbmQgYm9keS5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBpZ2F0ZXdheS9sYXRlc3QvZGV2ZWxvcGVyZ3VpZGUvYXBpLWdhdGV3YXktc3dhZ2dlci1leHRlbnNpb25zLWludGVncmF0aW9uLXJlcXVlc3RQYXJhbWV0ZXJzLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHJlcXVlc3RQYXJhbWV0ZXJzPzogeyBbcHJvcGVydHk6IHN0cmluZ106IHN0cmluZyB9O1xuICAvKipcbiAgICogTWFwcGluZyB0ZW1wbGF0ZXMgZm9yIGEgcmVxdWVzdCBwYXlsb2FkIG9mIHNwZWNpZmllZCBNSU1FIHR5cGVzLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1zd2FnZ2VyLWV4dGVuc2lvbnMtaW50ZWdyYXRpb24tcmVxdWVzdFRlbXBsYXRlcy5odG1sXG4gICAqL1xuICByZWFkb25seSByZXF1ZXN0VGVtcGxhdGVzPzogeyBbbWltZVR5cGU6IHN0cmluZ106IHN0cmluZyB9O1xuICAvKipcbiAgICogRGVmaW5lcyB0aGUgbWV0aG9kJ3MgcmVzcG9uc2VzIGFuZCBzcGVjaWZpZXMgZGVzaXJlZCBwYXJhbWV0ZXIgbWFwcGluZ3Mgb3IgcGF5bG9hZCBtYXBwaW5ncyBmcm9tIGludGVncmF0aW9uXG4gICAqIHJlc3BvbnNlcyB0byBtZXRob2QgcmVzcG9uc2VzLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1zd2FnZ2VyLWV4dGVuc2lvbnMtaW50ZWdyYXRpb24tcmVzcG9uc2VzLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHJlc3BvbnNlcz86IHtcbiAgICBbcmVzcG9uc2VTdGF0dXNQYXR0ZXJuOiBzdHJpbmddOiBBcGlHYXRld2F5SW50ZWdyYXRpb25SZXNwb25zZTtcbiAgfTtcbiAgLyoqXG4gICAqIEN1c3RvbSB0aW1lb3V0IGJldHdlZW4gNTAgYW5kIDI5LDAwMCBtaWxsaXNlY29uZHMuIFRoZSBkZWZhdWx0IHZhbHVlIGlzIDI5LDAwMCBtaWxsaXNlY29uZHMgb3IgMjkgc2Vjb25kcy5cbiAgICovXG4gIHJlYWRvbmx5IHRpbWVvdXRJbk1pbGxpcz86IG51bWJlcjtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyB0aGUgVExTIGNvbmZpZ3VyYXRpb24gZm9yIGFuIGludGVncmF0aW9uLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcGlnYXRld2F5L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hcGktZ2F0ZXdheS1leHRlbnNpb25zLWludGVncmF0aW9uLXRscy1jb25maWcuaHRtbFxuICAgKi9cbiAgcmVhZG9ubHkgdGxzQ29uZmlnPzogQXBpR2F0ZXdheUludGVncmF0aW9uVGxzQ29uZmlnO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIHJlbmRlcmluZyBhbiBpbnRlZ3JhdGlvbiBpbnRvIGFuIEFQSSBHYXRld2F5IE9wZW5BUEkgZXh0ZW5zaW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW50ZWdyYXRpb25SZW5kZXJQcm9wcyBleHRlbmRzIE1ldGhvZEFuZFBhdGgge1xuICAvKipcbiAgICogVGhlIElEIG9mIHRoZSBvcGVyYXRpb24gYmVpbmcgcmVuZGVyZWRcbiAgICovXG4gIHJlYWRvbmx5IG9wZXJhdGlvbklkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgc2NvcGUgaW4gd2hpY2ggdGhlIGludGVncmF0aW9uIGlzIGJlaW5nIHJlbmRlcmVkXG4gICAqL1xuICByZWFkb25seSBzY29wZTogQ29uc3RydWN0O1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGdyYW50aW5nIHRoZSBBUEkgYWNjZXNzIHRvIGludm9rZSB0aGUgb3BlcmF0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSW50ZWdyYXRpb25HcmFudFByb3BzIGV4dGVuZHMgTWV0aG9kQW5kUGF0aCB7XG4gIC8qKlxuICAgKiBUaGUgSUQgb2YgdGhlIG9wZXJhdGlvbiBmb3Igd2hpY2ggcGVybWlzc2lvbnMgYXJlIGJlaW5nIGdyYW50ZWRcbiAgICovXG4gIHJlYWRvbmx5IG9wZXJhdGlvbklkOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgc2NvcGUgaW4gd2hpY2ggcGVybWlzc2lvbiByZXNvdXJjZXMgY2FuIGJlIGNyZWF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHNjb3BlOiBDb25zdHJ1Y3Q7XG4gIC8qKlxuICAgKiBUaGUgYXBpIHRvIGdyYW50IHBlcm1pc3Npb25zIGZvclxuICAgKi9cbiAgcmVhZG9ubHkgYXBpOiBTcGVjUmVzdEFwaTtcbn1cblxuLyoqXG4gKiBBbiBpbnRlZ3JhdGlvbiBmb3IgYW4gQVBJIG9wZXJhdGlvbi4gWW91IGNhbiBleHRlbmQgdGhpcyB0byBpbXBsZW1lbnQgeW91ciBvd24gaW50ZWdyYXRpb24gaWYgeW91IGxpa2UuXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBJbnRlZ3JhdGlvbiB7XG4gIC8qKlxuICAgKiBSZW5kZXIgdGhlIGludGVncmF0aW9uIGludG8gYW4gQVBJIEdhdGV3YXkgT3BlbkFQSSBleHRlbnNpb25cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZW5kZXIocHJvcHM6IEludGVncmF0aW9uUmVuZGVyUHJvcHMpOiBBcGlHYXRld2F5SW50ZWdyYXRpb247XG4gIC8qKlxuICAgKiBHcmFudCBwZXJtaXNzaW9ucyBmb3IgdGhlIEFQSSB0byBpbnZva2UgdGhlIGludGVncmF0aW9uXG4gICAqL1xuICBwdWJsaWMgZ3JhbnQoX3Byb3BzOiBJbnRlZ3JhdGlvbkdyYW50UHJvcHMpIHtcbiAgICAvLyBObyBwZXJtaXNzaW9ucyB0byBncmFudCBieSBkZWZhdWx0XG4gIH1cbn1cblxuLyoqXG4gKiBBIGxhbWJkYSBpbnRlZ3JhdGlvblxuICovXG5leHBvcnQgY2xhc3MgTGFtYmRhSW50ZWdyYXRpb24gZXh0ZW5kcyBJbnRlZ3JhdGlvbiB7XG4gIHByaXZhdGUgcmVhZG9ubHkgbGFtYmRhRnVuY3Rpb246IElGdW5jdGlvbjtcblxuICBjb25zdHJ1Y3RvcihsYW1iZGFGdW5jdGlvbjogSUZ1bmN0aW9uKSB7XG4gICAgc3VwZXIoKTtcbiAgICB0aGlzLmxhbWJkYUZ1bmN0aW9uID0gbGFtYmRhRnVuY3Rpb247XG4gIH1cblxuICAvKipcbiAgICogUmVuZGVyIHRoZSBsYW1iZGEgaW50ZWdyYXRpb24gYXMgYSBzbmlwcGV0IG9mIE9wZW5BUElcbiAgICovXG4gIHB1YmxpYyByZW5kZXIocHJvcHM6IEludGVncmF0aW9uUmVuZGVyUHJvcHMpOiBBcGlHYXRld2F5SW50ZWdyYXRpb24ge1xuICAgIHJldHVybiB7XG4gICAgICB0eXBlOiBcIkFXU19QUk9YWVwiLFxuICAgICAgaHR0cE1ldGhvZDogXCJQT1NUXCIsXG4gICAgICB1cmk6IGZ1bmN0aW9uSW52b2NhdGlvblVyaShwcm9wcy5zY29wZSwgdGhpcy5sYW1iZGFGdW5jdGlvbiksXG4gICAgICBwYXNzdGhyb3VnaEJlaGF2aW9yOiBcIldIRU5fTk9fTUFUQ0hcIixcbiAgICB9O1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IEFQSSBHYXRld2F5IHBlcm1pc3Npb25zIHRvIGludm9rZSB0aGUgbGFtYmRhXG4gICAqL1xuICBwdWJsaWMgZ3JhbnQoe1xuICAgIHNjb3BlLFxuICAgIGFwaSxcbiAgICBvcGVyYXRpb25JZCxcbiAgICBtZXRob2QsXG4gICAgcGF0aCxcbiAgfTogSW50ZWdyYXRpb25HcmFudFByb3BzKSB7XG4gICAgbmV3IENmblBlcm1pc3Npb24oc2NvcGUsIGBMYW1iZGFQZXJtaXNzaW9uLSR7b3BlcmF0aW9uSWR9YCwge1xuICAgICAgYWN0aW9uOiBcImxhbWJkYTpJbnZva2VGdW5jdGlvblwiLFxuICAgICAgcHJpbmNpcGFsOiBcImFwaWdhdGV3YXkuYW1hem9uYXdzLmNvbVwiLFxuICAgICAgZnVuY3Rpb25OYW1lOiB0aGlzLmxhbWJkYUZ1bmN0aW9uLmZ1bmN0aW9uQXJuLFxuICAgICAgc291cmNlQXJuOiBTdGFjay5vZihzY29wZSkuZm9ybWF0QXJuKHtcbiAgICAgICAgc2VydmljZTogXCJleGVjdXRlLWFwaVwiLFxuICAgICAgICByZXNvdXJjZTogYXBpLnJlc3RBcGlJZCxcbiAgICAgICAgLy8gU2NvcGUgcGVybWlzc2lvbnMgdG8gdGhlIHN0YWdlLCBtZXRob2QgYW5kIHBhdGggb2YgdGhlIG9wZXJhdGlvbi5cbiAgICAgICAgLy8gUGF0aCBwYXJhbWV0ZXJzIChlZyB7cGFyYW19IGFyZSByZXBsYWNlZCB3aXRoIHdpbGRjYXJkcylcbiAgICAgICAgcmVzb3VyY2VOYW1lOiBgJHtcbiAgICAgICAgICBhcGkuZGVwbG95bWVudFN0YWdlLnN0YWdlTmFtZVxuICAgICAgICB9LyR7bWV0aG9kLnRvVXBwZXJDYXNlKCl9JHtwYXRoLnJlcGxhY2UoL3tbXlxcfV0qXFx9L2csIFwiKlwiKX1gLFxuICAgICAgfSksXG4gICAgfSk7XG4gIH1cbn1cblxuLyoqXG4gKiBBIGNvbGxlY3Rpb24gb2YgaW50ZWdyYXRpb25zIHRvIGNvbm5lY3QgQVBJIG9wZXJhdGlvbnMgd2l0aCBhIGJhY2tlbmQgdG8gc2VydmljZSByZXF1ZXN0c1xuICovXG5leHBvcnQgY2xhc3MgSW50ZWdyYXRpb25zIHtcbiAgLyoqXG4gICAqIEFuIGludGVncmF0aW9uIHRoYXQgaW52b2tlcyBhIGxhbWJkYSBmdW5jdGlvbiB0byBzZXJ2aWNlIHRoZSByZXF1ZXN0XG4gICAqIEBwYXJhbSBsYW1iZGFGdW5jdGlvbiB0aGUgZnVuY3Rpb24gdG8gaW52b2tlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGxhbWJkYShsYW1iZGFGdW5jdGlvbjogSUZ1bmN0aW9uKTogTGFtYmRhSW50ZWdyYXRpb24ge1xuICAgIHJldHVybiBuZXcgTGFtYmRhSW50ZWdyYXRpb24obGFtYmRhRnVuY3Rpb24pO1xuICB9XG59XG4iXX0=