from .utils import date, run_sftp, test_access, to_path
from inform import (
    Error, codicil, comment, cull, error, fmt, is_str, narrate, os_error, warn
)


class AuthKeys:
    known = {}
    def __new__(cls, server, include_file, bypass, trial_run):
        if server in AuthKeys.known:
            self = AuthKeys.known[server]
            if include_file != self.include_file:
                warn(
                    'inconsistent remote include file:',
                    fmt('{include_file} != {self.include_file} in {server}.')
                )
            return self
        self = super(AuthKeys, cls).__new__(cls)
        AuthKeys.known[server] = self
        self.server = server
        self.bypass = bypass
        self.trial_run = trial_run
        self.keys = {}
        self.comment = {}
        self.restrictions = {}
        self.include_file = include_file
        self.include = None

        # get remote include file if it exists
        if include_file and not bypass:
            narrate(fmt('    retrieving remote include file from {server}.'))
            try:
                run_sftp(self.server, [
                    fmt('get .ssh/{inc} {inc}.{server}', inc=include_file)
                ])
                self.include = to_path(include_file + '.' + server).read_text()
            except Error as e:
                comment(fmt('    sftp {server}: {include_file} not found.'))
                codicil(e)

        return self

    def add_public_key(self, name, key, comment, restrictions):
        assert name not in self.keys
        self.keys[name] = key
        self.comment[name] = comment
        self.restrictions[name] = restrictions

    def publish(self):
        narrate('publishing authorized_keys to', self.server)
        prov = '.provisional' if self.trial_run else ''
        entries = [
            fmt("# This file was generated by sshdeploy on {date}.")
        ]
        if self.include:
            entries += [
                '\n'.join([
                    fmt('# Contents of {self.include_file}:'),
                    self.include
                ])
            ]
        for name in sorted(self.keys.keys()):
            key = self.keys[name]
            comment = self.comment[name]
            comment = [comment] if is_str(comment) else comment
            restrictions = self.restrictions[name]
            if not is_str(restrictions):
                restrictions = ','.join(restrictions)
            restricted_key = ' '.join(cull([restrictions, key]))
            entries.append('\n'.join(comment + [restricted_key]))

        # delete any pre-existing provisional files
        # the goal here is to leave a clean directory when not trial-run
        try:
            run_sftp(self.server, [
                fmt('rm .ssh/*.provisional')
            ])
        except Error:
            pass

        # now upload the new authorized_keys file
        try:
            authkey = to_path('authorized_keys.%s' % self.server)
            with authkey.open('w') as f:
                f.write('\n\n'.join(entries) + '\n')
            authkey.chmod(0o600)
            if self.bypass:
                warn(
                    'You must manually upload',
                    fmt('<keydir>/authorized_keys.{self.server}.'),
                    culprit=self.server
                )
            else:
                cmds = [fmt('put -p {authkey} .ssh/authorized_keys{prov}')]
                for name in self.keys:
                    cmds.append(fmt('put -p {name}.pub .ssh/{name}.pub{prov}'))
                run_sftp(self.server, cmds)
        except Error as e:
            error(e)

    def verify(self):
        if self.bypass:
            narrate('skipping post update connection test for', self.server)
            return

        comment('%s: post update connection test.' % (self.server))
        test_access(self.server)
