"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateClientProjects = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const languages_1 = require("../languages");
const generated_python_client_1 = require("./generated-python-client");
const generated_typescript_client_project_1 = require("./generated-typescript-client-project");
/**
 * Returns a generated client project for the given language
 */
const generateClientProject = (language, options) => {
    switch (language) {
        case languages_1.ClientLanguage.TYPESCRIPT:
            return new generated_typescript_client_project_1.GeneratedTypescriptClientProject({
                parent: options.parent,
                // Ensure kebab-case for typescript
                name: `${options.parentPackageName}-${languages_1.ClientLanguage.TYPESCRIPT}`.replace(/_/g, "-"),
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.TYPESCRIPT),
                specPath: options.parsedSpecPath,
                ...options.typescriptOptions,
            });
        case languages_1.ClientLanguage.PYTHON:
            // Ensure snake_case for python
            const name = `${options.parentPackageName}_${languages_1.ClientLanguage.PYTHON}`
                .replace(/@/g, "")
                .replace(/[\-/]/g, "_");
            return new generated_python_client_1.GeneratedPythonClientProject({
                parent: options.parent,
                rootProjectHasParent: options.rootProjectHasParent,
                name,
                moduleName: name,
                outdir: path.join(options.generatedCodeDir, languages_1.ClientLanguage.PYTHON),
                specPath: options.parsedSpecPath,
                ...options.pythonOptions,
            });
        default:
            throw new Error(`Unknown client language ${language}`);
    }
};
/**
 * Generate API clients in the given languages
 * @param languages the languages to generate clients for
 * @param options options for the projects to be created
 */
exports.generateClientProjects = (languages, options) => Object.fromEntries([...languages].map((language) => [
    language,
    generateClientProject(language, options),
]));
//# sourceMappingURL=data:application/json;base64,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