"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends core_1.Construct {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        const createKmsKey = (_a = props.createKmsKey) !== null && _a !== void 0 ? _a : true;
        let encryptionAlias;
        if (createKmsKey) {
            const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
                enableKeyRotation: props.enableKeyRotation,
            });
            encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
                targetKey: encryptionKey,
                aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
        }
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryption: encryptionAlias ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default', {
            createKmsKey: props.createKmsKey,
            enableKeyRotation: props.enableKeyRotation,
        });
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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