"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToStepFunction = void 0;
const defaults = require("@aws-solutions-constructs/core");
const aws_events_rule_step_function_1 = require("@aws-solutions-constructs/aws-events-rule-step-function");
const core_1 = require("@aws-cdk/core");
const cloudtrail = require("@aws-cdk/aws-cloudtrail");
class S3ToStepFunction extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the S3ToStepFunction class.
     * @since 0.9.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        let bucket;
        if (!props.existingBucketObj) {
            [this.s3Bucket, this.s3LoggingBucket] = defaults.buildS3Bucket(this, {
                bucketProps: props.bucketProps
            });
            bucket = this.s3Bucket;
        }
        else {
            bucket = props.existingBucketObj;
        }
        if (props.deployCloudTrail === undefined || props.deployCloudTrail) {
            [this.cloudtrailBucket, this.cloudtrailLoggingBucket] = defaults.buildS3Bucket(this, {}, 'CloudTrail');
            this.cloudtrail = new cloudtrail.Trail(this, 'S3EventsTrail', {
                bucket: this.cloudtrailBucket
            });
            this.cloudtrail.addS3EventSelector([{
                    bucket
                }], {
                readWriteType: cloudtrail.ReadWriteType.ALL,
                includeManagementEvents: false
            });
        }
        let _eventRuleProps = {};
        if (props.eventRuleProps) {
            _eventRuleProps = props.eventRuleProps;
        }
        else {
            // By default the CW Events Rule will filter any 's3:PutObject' events for the S3 Bucket
            _eventRuleProps = {
                eventPattern: {
                    source: ['aws.s3'],
                    detailType: ['AWS API Call via CloudTrail'],
                    detail: {
                        eventSource: [
                            "s3.amazonaws.com"
                        ],
                        eventName: [
                            "PutObject"
                        ],
                        requestParameters: {
                            bucketName: [
                                bucket.bucketName
                            ]
                        }
                    }
                }
            };
        }
        const eventsRuleToStepFunction = new aws_events_rule_step_function_1.EventsRuleToStepFunction(this, 'test-events-rule-step-function-stack', {
            stateMachineProps: props.stateMachineProps,
            eventRuleProps: _eventRuleProps,
            createCloudWatchAlarms: props.createCloudWatchAlarms,
            logGroupProps: props.logGroupProps
        });
        this.stateMachine = eventsRuleToStepFunction.stateMachine;
        this.stateMachineLogGroup = eventsRuleToStepFunction.stateMachineLogGroup;
        this.cloudwatchAlarms = eventsRuleToStepFunction.cloudwatchAlarms;
    }
}
exports.S3ToStepFunction = S3ToStepFunction;
//# sourceMappingURL=data:application/json;base64,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