"use strict";
/*
 * Copyright 2018-2021 Elyra Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const ui_components_1 = require("@elyra/ui-components");
const application_1 = require("@jupyterlab/application");
const apputils_1 = require("@jupyterlab/apputils");
const launcher_1 = require("@jupyterlab/launcher");
const mainmenu_1 = require("@jupyterlab/mainmenu");
const ui_components_2 = require("@jupyterlab/ui-components");
const algorithm_1 = require("@lumino/algorithm");
const launcher_2 = require("./launcher");
require("../style/index.css");
const ELYRA_THEME_NAMESPACE = 'elyra-theme-extension';
/**
 * The command IDs used by the launcher plugin.
 */
const CommandIDs = {
    create: 'launcher:create',
    openHelp: 'elyra:open-help'
};
/**
 * Initialization data for the theme extension.
 */
const extension = {
    id: ELYRA_THEME_NAMESPACE,
    autoStart: true,
    requires: [application_1.ILabShell, mainmenu_1.IMainMenu],
    optional: [apputils_1.ICommandPalette],
    provides: launcher_1.ILauncher,
    activate: (app, labShell, mainMenu, palette) => {
        console.log('Elyra - theme extension is activated!');
        // Find the MainLogo widget in the shell and replace it with the Elyra Logo
        const widgets = app.shell.widgets('top');
        let widget = widgets.next();
        while (widget !== undefined) {
            if (widget.id === 'jp-MainLogo') {
                ui_components_1.elyraIcon.element({
                    container: widget.node,
                    justify: 'center',
                    margin: '2px 5px 2px 5px',
                    height: 'auto',
                    width: '20px'
                });
                break;
            }
            widget = widgets.next();
        }
        // Use custom Elyra launcher
        const { commands } = app;
        const model = new launcher_1.LauncherModel();
        commands.addCommand(CommandIDs.create, {
            label: 'New Launcher',
            execute: (args) => {
                const cwd = args['cwd'] ? String(args['cwd']) : '';
                const id = `launcher-${Private.id++}`;
                const callback = (item) => {
                    labShell.add(item, 'main', { ref: id });
                };
                const launcher = new launcher_2.Launcher({ model, cwd, callback, commands });
                launcher.model = model;
                launcher.title.icon = ui_components_2.launcherIcon;
                launcher.title.label = 'Launcher';
                const main = new apputils_1.MainAreaWidget({ content: launcher });
                // If there are any other widgets open, remove the launcher close icon.
                main.title.closable = !!algorithm_1.toArray(labShell.widgets('main')).length;
                main.id = id;
                labShell.add(main, 'main', { activate: args['activate'] });
                labShell.layoutModified.connect(() => {
                    // If there is only a launcher open, remove the close icon.
                    main.title.closable = algorithm_1.toArray(labShell.widgets('main')).length > 1;
                }, main);
                return main;
            }
        });
        if (palette) {
            palette.addItem({ command: CommandIDs.create, category: 'Launcher' });
        }
        commands.addCommand(CommandIDs.openHelp, {
            label: 'Documentation',
            icon: ui_components_1.helpIcon,
            execute: (args) => {
                window.open('https://elyra.readthedocs.io/en/v3.0.0/', '_blank');
            }
        });
        model.add({
            command: CommandIDs.openHelp,
            category: 'Elyra',
            rank: 10
        });
        return model;
    }
};
/**
 * The namespace for module private data.
 */
var Private;
(function (Private) {
    /**
     * The incrementing id used for launcher widgets.
     */
    // eslint-disable-next-line
    Private.id = 0;
})(Private || (Private = {}));
exports.default = extension;
//# sourceMappingURL=index.js.map