"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const url_1 = require("url");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * (experimental) Elasticsearch version.
 *
 * @experimental
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * (experimental) Custom Elasticsearch version.
     *
     * @param version custom version number.
     * @experimental
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
/**
 * (experimental) AWS Elasticsearch 1.5.
 *
 * @experimental
 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/**
 * (experimental) AWS Elasticsearch 2.3.
 *
 * @experimental
 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/**
 * (experimental) AWS Elasticsearch 5.1.
 *
 * @experimental
 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/**
 * (experimental) AWS Elasticsearch 5.3.
 *
 * @experimental
 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/**
 * (experimental) AWS Elasticsearch 5.5.
 *
 * @experimental
 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/**
 * (experimental) AWS Elasticsearch 5.6.
 *
 * @experimental
 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/**
 * (experimental) AWS Elasticsearch 6.0.
 *
 * @experimental
 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/**
 * (experimental) AWS Elasticsearch 6.2.
 *
 * @experimental
 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/**
 * (experimental) AWS Elasticsearch 6.3.
 *
 * @experimental
 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/**
 * (experimental) AWS Elasticsearch 6.4.
 *
 * @experimental
 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/**
 * (experimental) AWS Elasticsearch 6.5.
 *
 * @experimental
 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/**
 * (experimental) AWS Elasticsearch 6.7.
 *
 * @experimental
 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/**
 * (experimental) AWS Elasticsearch 6.8.
 *
 * @experimental
 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/**
 * (experimental) AWS Elasticsearch 7.1.
 *
 * @experimental
 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/**
 * (experimental) AWS Elasticsearch 7.4.
 *
 * @experimental
 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/**
 * (experimental) AWS Elasticsearch 7.7.
 *
 * @experimental
 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * (experimental) AWS Elasticsearch 7.8.
 *
 * @experimental
 */
ElasticsearchVersion.V7_8 = ElasticsearchVersion.of('7.8');
/**
 * (experimental) AWS Elasticsearch 7.9.
 *
 * @experimental
 */
ElasticsearchVersion.V7_9 = ElasticsearchVersion.of('7.9');
/**
 * (experimental) The minimum TLS version required for traffic to the domain.
 *
 * @experimental
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * (experimental) Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Return the given named metric for this Domain.
     *
     * @experimental
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensions: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * (experimental) Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @experimental
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @experimental
     */
    metricClusterIndexWriteBlocked(props) {
        return this.metric('ClusterIndexWriteBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @experimental
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatency', { statistic: 'p99', ...props });
    }
    /**
     * (experimental) Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatency', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * (experimental) Provides an Elasticsearch domain.
 *
 * @experimental
 */
class Domain extends DomainBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29, _30, _31, _32, _33, _34, _35, _36;
        super(scope, id, {
            physicalName: props.domainName,
        });
        const defaultInstanceType = 'r5.large.elasticsearch';
        const warmDefaultInstanceType = 'ultrawarm1.medium.elasticsearch';
        const dedicatedMasterType = (_c = (_b = (_a = props.capacity) === null || _a === void 0 ? void 0 : _a.masterNodeInstanceType) === null || _b === void 0 ? void 0 : _b.toLowerCase()) !== null && _c !== void 0 ? _c : defaultInstanceType;
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = dedicatedMasterCount > 0;
        const instanceType = (_h = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType) === null || _g === void 0 ? void 0 : _g.toLowerCase()) !== null && _h !== void 0 ? _h : defaultInstanceType;
        const instanceCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.dataNodes) !== null && _k !== void 0 ? _k : 1;
        const warmType = (_o = (_m = (_l = props.capacity) === null || _l === void 0 ? void 0 : _l.warmInstanceType) === null || _m === void 0 ? void 0 : _m.toLowerCase()) !== null && _o !== void 0 ? _o : warmDefaultInstanceType;
        const warmCount = (_q = (_p = props.capacity) === null || _p === void 0 ? void 0 : _p.warmNodes) !== null && _q !== void 0 ? _q : 0;
        const warmEnabled = warmCount > 0;
        const availabilityZoneCount = (_s = (_r = props.zoneAwareness) === null || _r === void 0 ? void 0 : _r.availabilityZoneCount) !== null && _s !== void 0 ? _s : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_u = (_t = props.zoneAwareness) === null || _t === void 0 ? void 0 : _t.enabled) !== null && _u !== void 0 ? _u : ((_v = props.zoneAwareness) === null || _v === void 0 ? void 0 : _v.availabilityZoneCount) != null;
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (props.vpcOptions != null && zoneAwarenessEnabled &&
            new Set((_w = props.vpcOptions) === null || _w === void 0 ? void 0 : _w.subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        ;
        if ([dedicatedMasterType, instanceType, warmType].some(t => !t.endsWith('.elasticsearch'))) {
            throw new Error('Master, data and UltraWarm node instance types must end with ".elasticsearch".');
        }
        if (!warmType.startsWith('ultrawarm')) {
            throw new Error('UltraWarm node instance type must start with "ultrawarm".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_x = props.useUnsignedBasicAuth) !== null && _x !== void 0 ? _x : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_y = props.encryptionAtRest) === null || _y === void 0 ? void 0 : _y.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.Anyone()],
            resources: [cdk.Lazy.string({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_z = props.fineGrainedAccessControl) === null || _z === void 0 ? void 0 : _z.masterUserArn;
        const masterUserNameProps = (_0 = props.fineGrainedAccessControl) === null || _0 === void 0 ? void 0 : _0.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_1 = props.fineGrainedAccessControl) === null || _1 === void 0 ? void 0 : _1.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_3 = (_2 = props.encryptionAtRest) === null || _2 === void 0 ? void 0 : _2.enabled) !== null && _3 !== void 0 ? _3 : (((_4 = props.encryptionAtRest) === null || _4 === void 0 ? void 0 : _4.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_5 = props.nodeToNodeEncryption) !== null && _5 !== void 0 ? _5 : unsignedBasicAuthEnabled;
        const volumeSize = (_7 = (_6 = props.ebs) === null || _6 === void 0 ? void 0 : _6.volumeSize) !== null && _7 !== void 0 ? _7 : 10;
        const volumeType = (_9 = (_8 = props.ebs) === null || _8 === void 0 ? void 0 : _8.volumeType) !== null && _9 !== void 0 ? _9 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_11 = (_10 = props.ebs) === null || _10 === void 0 ? void 0 : _10.enabled) !== null && _11 !== void 0 ? _11 : true;
        const enforceHttps = (_12 = props.enforceHttps) !== null && _12 !== void 0 ? _12 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if (((_13 = props.logging) === null || _13 === void 0 ? void 0 : _13.slowIndexLogEnabled) || ((_14 = props.logging) === null || _14 === void 0 ? void 0 : _14.appLogEnabled)
                || ((_15 = props.logging) === null || _15 === void 0 ? void 0 : _15.slowSearchLogEnabled)) {
                throw new Error('Error and slow logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_16 = props.encryptionAtRest) === null || _16 === void 0 ? void 0 : _16.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.8 && warmEnabled) {
            throw new Error('UltraWarm requires Elasticsearch 6.8 or later.');
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isInstanceType('i3') && ebsEnabled) {
            throw new Error('I3 instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        if (isSomeInstanceType('t2', 't3') && warmEnabled) {
            throw new Error('T2 and T3 instance types do not support UltraWarm storage.');
        }
        // Only R3 and I3 support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3')) {
            throw new Error('EBS volumes are required when using instance types other than r3 or i3.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        // Validate fine grained access control enabled for audit logs, per
        // https://aws.amazon.com/about-aws/whats-new/2020/09/elasticsearch-audit-logs-now-available-on-amazon-elasticsearch-service/
        if (((_17 = props.logging) === null || _17 === void 0 ? void 0 : _17.auditLogEnabled) && !advancedSecurityEnabled) {
            throw new Error('Fine-grained access control is required when audit logs publishing is enabled.');
        }
        // Validate UltraWarm requirement for dedicated master nodes, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/ultrawarm.html
        if (warmEnabled && !dedicatedMasterEnabled) {
            throw new Error('Dedicated master node is required when UltraWarm storage is enabled.');
        }
        let cfnVpcOptions;
        if (props.vpcOptions) {
            cfnVpcOptions = {
                securityGroupIds: props.vpcOptions.securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: props.vpcOptions.subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_18 = props.logging) === null || _18 === void 0 ? void 0 : _18.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_19 = props.logging.slowSearchLogGroup) !== null && _19 !== void 0 ? _19 : new logs.LogGroup(this, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_20 = props.logging) === null || _20 === void 0 ? void 0 : _20.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_21 = props.logging.slowIndexLogGroup) !== null && _21 !== void 0 ? _21 : new logs.LogGroup(this, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_22 = props.logging) === null || _22 === void 0 ? void 0 : _22.appLogEnabled) {
            this.appLogGroup = (_23 = props.logging.appLogGroup) !== null && _23 !== void 0 ? _23 : new logs.LogGroup(this, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        if ((_24 = props.logging) === null || _24 === void 0 ? void 0 : _24.auditLogEnabled) {
            this.auditLogGroup = (_25 = props.logging.auditLogGroup) !== null && _25 !== void 0 ? _25 : new logs.LogGroup(this, 'AuditLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.auditLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, `ESLogGroupPolicy${this.node.addr}`, {
                // create a cloudwatch logs resource policy name that is unique to this domain instance
                policyName: `ESLogPolicy${this.node.addr}`,
                policyStatements: [logPolicyStatement],
            });
        }
        const logPublishing = {};
        if (this.appLogGroup) {
            logPublishing.ES_APPLICATION_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.appLogGroup.logGroupArn,
            };
        }
        if (this.slowSearchLogGroup) {
            logPublishing.SEARCH_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowSearchLogGroup.logGroupArn,
            };
        }
        if (this.slowIndexLogGroup) {
            logPublishing.INDEX_SLOW_LOGS = {
                enabled: true,
                cloudWatchLogsLogGroupArn: this.slowIndexLogGroup.logGroupArn,
            };
        }
        if (this.auditLogGroup) {
            logPublishing.AUDIT_LOGS = {
                enabled: this.auditLogGroup != null,
                cloudWatchLogsLogGroupArn: (_26 = this.auditLogGroup) === null || _26 === void 0 ? void 0 : _26.logGroupArn,
            };
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                warmEnabled: warmEnabled
                    ? warmEnabled
                    : undefined,
                warmCount: warmEnabled
                    ? warmCount
                    : undefined,
                warmType: warmEnabled
                    ? warmType
                    : undefined,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_27 = props.ebs) === null || _27 === void 0 ? void 0 : _27.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_29 = (_28 = props.encryptionAtRest) === null || _28 === void 0 ? void 0 : _28.kmsKey) === null || _29 === void 0 ? void 0 : _29.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: logPublishing,
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_30 = props.cognitoKibanaAuth) === null || _30 === void 0 ? void 0 : _30.identityPoolId,
                roleArn: (_31 = props.cognitoKibanaAuth) === null || _31 === void 0 ? void 0 : _31.role.roleArn,
                userPoolId: (_32 = props.cognitoKibanaAuth) === null || _32 === void 0 ? void 0 : _32.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_33 = props.tlsSecurityPolicy) !== null && _33 !== void 0 ? _33 : TLSSecurityPolicy.TLS_1_0,
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_34 = this.masterUserPassword) === null || _34 === void 0 ? void 0 : _34.toString(),
                    },
                }
                : undefined,
        });
        if (props.enableVersionUpgrade) {
            this.domain.cfnOptions.updatePolicy = {
                ...this.domain.cfnOptions.updatePolicy,
                enableVersionUpgrade: props.enableVersionUpgrade,
            };
        }
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_35 = props.accessPolicies) !== null && _35 !== void 0 ? _35 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            if ((_36 = props.encryptionAtRest) === null || _36 === void 0 ? void 0 : _36.kmsKey) {
                // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/encryption-at-rest.html
                // these permissions are documented as required during domain creation.
                // while not strictly documented for updates as well, it stands to reason that an update
                // operation might require these in case the cluster uses a kms key.
                // empircal evidence shows this is indeed required: https://github.com/aws/aws-cdk/issues/11412
                accessPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                    actions: ['kms:List*', 'kms:Describe*', 'kms:CreateGrant'],
                    resources: [props.encryptionAtRest.kmsKey.keyArn],
                    effect: iam.Effect.ALLOW,
                }));
            }
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @experimental
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @experimental
     */
    static fromDomainAttributes(scope, id, attrs) {
        const { domainArn, domainEndpoint } = attrs;
        const domainName = extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
}
exports.Domain = Domain;
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
//# sourceMappingURL=data:application/json;base64,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