"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArtifactPath = exports.Artifact = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const validation = require("./private/validation");
/**
 * An output artifact of an action. Artifacts can be used as input by some actions.
 */
class Artifact {
    constructor(artifactName) {
        this.metadata = {};
        validation.validateArtifactName(artifactName);
        this._artifactName = artifactName;
    }
    /**
     * A static factory method used to create instances of the Artifact class.
     * Mainly meant to be used from `decdk`.
     *
     * @param name the (required) name of the Artifact
     */
    static artifact(name) {
        return new Artifact(name);
    }
    get artifactName() {
        return this._artifactName;
    }
    /**
     * Returns an ArtifactPath for a file within this artifact.
     * CfnOutput is in the form "<artifact-name>::<file-name>"
     * @param fileName The name of the file
     */
    atPath(fileName) {
        return new ArtifactPath(this, fileName);
    }
    /**
     * The artifact attribute for the name of the S3 bucket where the artifact is stored.
     */
    get bucketName() {
        return artifactAttribute(this, 'BucketName');
    }
    /**
     * The artifact attribute for The name of the .zip file that contains the artifact that is
     * generated by AWS CodePipeline, such as 1ABCyZZ.zip.
     */
    get objectKey() {
        return artifactAttribute(this, 'ObjectKey');
    }
    /**
     * The artifact attribute of the Amazon Simple Storage Service (Amazon S3) URL of the artifact,
     * such as https://s3-us-west-2.amazonaws.com/artifactstorebucket-yivczw8jma0c/test/TemplateSo/1ABCyZZ.zip.
     */
    get url() {
        return artifactAttribute(this, 'URL');
    }
    /**
     * Returns a token for a value inside a JSON file within this artifact.
     * @param jsonFile The JSON file name.
     * @param keyName The hash key.
     */
    getParam(jsonFile, keyName) {
        return artifactGetParam(this, jsonFile, keyName);
    }
    /**
     * Returns the location of the .zip file in S3 that this Artifact represents.
     * Used by Lambda's `CfnParametersCode` when being deployed in a CodePipeline.
     */
    get s3Location() {
        return {
            bucketName: this.bucketName,
            objectKey: this.objectKey,
        };
    }
    /**
     * Add arbitrary extra payload to the artifact under a given key.
     * This can be used by CodePipeline actions to communicate data between themselves.
     * If metadata was already present under the given key,
     * it will be overwritten with the new value.
     */
    setMetadata(key, value) {
        this.metadata[key] = value;
    }
    /**
     * Retrieve the metadata stored in this artifact under the given key.
     * If there is no metadata stored under the given key,
     * null will be returned.
     */
    getMetadata(key) {
        return this.metadata[key];
    }
    toString() {
        return this.artifactName;
    }
    /** @internal */
    _setName(name) {
        if (this._artifactName) {
            throw new Error(`Artifact already has name '${this._artifactName}', cannot override it`);
        }
        else {
            this._artifactName = name;
        }
    }
}
exports.Artifact = Artifact;
_a = JSII_RTTI_SYMBOL_1;
Artifact[_a] = { fqn: "@aws-cdk/aws-codepipeline.Artifact", version: "1.148.0" };
/**
 * A specific file within an output artifact.
 *
 * The most common use case for this is specifying the template file
 * for a CloudFormation action.
 */
class ArtifactPath {
    constructor(artifact, fileName) {
        this.artifact = artifact;
        this.fileName = fileName;
        jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_Artifact(artifact);
    }
    static artifactPath(artifactName, fileName) {
        return new ArtifactPath(Artifact.artifact(artifactName), fileName);
    }
    get location() {
        const artifactName = this.artifact.artifactName
            ? this.artifact.artifactName
            : core_1.Lazy.string({ produce: () => this.artifact.artifactName });
        return `${artifactName}::${this.fileName}`;
    }
}
exports.ArtifactPath = ArtifactPath;
_b = JSII_RTTI_SYMBOL_1;
ArtifactPath[_b] = { fqn: "@aws-cdk/aws-codepipeline.ArtifactPath", version: "1.148.0" };
function artifactAttribute(artifact, attributeName) {
    const lazyArtifactName = core_1.Lazy.string({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetArtifactAtt': [lazyArtifactName, attributeName] });
}
function artifactGetParam(artifact, jsonFile, keyName) {
    const lazyArtifactName = core_1.Lazy.string({ produce: () => artifact.artifactName });
    return core_1.Token.asString({ 'Fn::GetParam': [lazyArtifactName, jsonFile, keyName] });
}
//# sourceMappingURL=data:application/json;base64,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