#!/usr/bin/env python3
# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-

# Copyright (C) 2022 Authors
#
# Released under GNU GPLv2 or later, read the file 'LICENSE.GPLv2+' for
# more information.
#
# Authors:
#   Bryce Harrington <bryce@canonical.com>

"""An individual autopkgtest run"""

from typing import Iterator
import json

from .constants import URL_AUTOPKGTEST


class Job:
    """An individual autopkgtest run that has not yet completed.

    A Job will correspond to one Result object once it has completed.
    """
    def __init__(self, number, submit_time, source_package, series, arch,
                 triggers=None, ppas=None):
        """Initializes a new Job object.

        :param str number: Position within the waiting queue.
        :param str submit_time: Timestamp when job was submitted.
        :param str source_package: Source package containing the DEP8 tests to run.
        :param str series: Codename of the Ubuntu release to run tests on.
        :param str arch: Hardware architecture type to run tests on.
        :param list[str] triggers: List of package/version triggers for the job.
        :param list[str] ppas: List of PPAs to enable.
        """
        self.number = number
        self.submit_time = submit_time
        self.source_package = source_package
        self.series = series
        self.arch = arch
        self.triggers = triggers or []
        self.ppas = ppas or []

    def __repr__(self) -> str:
        """Machine-parsable unique representation of object.

        :rtype: str
        :returns: Official string representation of the object.
        """
        return (f'{self.__class__.__name__}('
                f'source_package={self.source_package!r}, '
                f'series={self.series!r}, '
                f'arch={self.arch!r}'
                f')')

    def __str__(self) -> str:
        """Human-readable summary of the object.

        :rtype: str
        :returns: Printable summary of the object.
        """
        return f"{self.source_package} {self.series} ({self.arch})"

    @property
    def request_url(self) -> str:
        """Renders URL for requesting the testing run be started

        :rtype: str
        :returns: Full URL for invoking the test.
        """
        rel_str = f"release={self.series}"
        arch_str = f"&arch={self.arch}"
        pkg_str = f"&package={self.source_package}"
        trigger_str = ''
        for trigger in self.triggers:
            trigger_str += f"&trigger={trigger}"
        return f"{URL_AUTOPKGTEST}/request.cgi?{rel_str}{arch_str}{pkg_str}{trigger_str}"


def get_running(response, series=None, ppa=None) -> Iterator[Job]:
    """Returns iterator currently running autopkgtests for given criteria

    Filters the list of running autopkgtest jobs by the given series
    and/or ppa names, returning an iterator with matching results as Job
    objects.  If series and ppa are not provided, then returns all
    results; if one or the other is provided, provides all available
    results for that series or ppa.

    :param HTTPResponse response: Context manager; the response from urlopen()
    :param str series: The Ubuntu release codename criteria, or None.
    :param str ppa: The PPA address criteria, or None.
    :rtype: Iterator[Job]
    :returns: Currently running jobs, if any, or an empty list on error
    """
    for pkg, jobs in json.loads(response.read().decode('utf-8')).items():
        for handle in jobs:
            for codename in jobs[handle]:
                for arch, jobinfo in jobs[handle][codename].items():
                    triggers = jobinfo[0].get('triggers', None)
                    ppas = jobinfo[0].get('ppas', None)
                    submit_time = jobinfo[1]
                    job = Job(0, submit_time, pkg, codename, arch, triggers, ppas)
                    if (series and (series != job.series)) or (ppa and (ppa not in job.ppas)):
                        continue
                    yield job


def get_waiting(response, series=None, ppa=None) -> Iterator[Job]:
    """Returns iterator of queued autopkgtests for given criteria

    Filters the list of autopkgtest jobs waiting for execution by the
    given series and/or ppa names, returning an iterator with matching
    results as Job objects.  If series and ppa are not provided, then
    returns all results; if one or the other is provided, provides all
    available results for that series or ppa.

    :param HTTPResponse response: Context manager; the response from urlopen()
    :param str series: The Ubuntu release codename criteria, or None.
    :param str ppa: The PPA address criteria, or None.
    :rtype: Iterator[Job]
    :returns: Currently waiting jobs, if any, or an empty list on error
    """
    for _, queue in json.loads(response.read().decode('utf-8')).items():
        for codename in queue:
            for arch in queue[codename]:
                n = 0
                for key in queue[codename][arch]:
                    if key == 'private job':
                        continue
                    (pkg, json_data) = key.split(maxsplit=1)
                    jobinfo = json.loads(json_data)
                    n += 1
                    triggers = jobinfo.get('triggers', None)
                    ppas = jobinfo.get('ppas', None)
                    job = Job(n, None, pkg, codename, arch, triggers, ppas)
                    if (series and (series != job.series)) or (ppa and (ppa not in job.ppas)):
                        continue
                    yield job


def show_running(jobs):
    """Prints the active (running and waiting) tests"""
    rformat = "    %-8s %-40s %-8s %-8s %-40s %s"

    n = 0
    for n, e in enumerate(jobs, start=1):
        if n == 1:
            print("Running:")
            ppa_str = ','.join(e.ppas)
            trig_str = ','.join(e.triggers)
            print(rformat % ("time", "pkg", "release", "arch", "ppa", "trigger"))
        print(rformat % (str(e.submit_time), e.source_package, e.series, e.arch, ppa_str, trig_str))
    if n == 0:
        print("Running: (none)")


def show_waiting(jobs):
    """Prints the active (running and waiting) tests"""
    rformat = "    %-8s %-40s %-8s %-8s %-40s %s"

    n = 0
    for n, e in enumerate(jobs, start=1):
        if n == 1:
            print("Waiting:")
            print(rformat % ("Q-num", "pkg", "release", "arch", "ppa", "trigger"))

        ppa_str = ','.join(e.ppas)
        trig_str = ','.join(e.triggers)
        print(rformat % (e.number, e.source_package, e.series, e.arch, ppa_str, trig_str))
    if n == 0:
        print("Waiting: (none)")


if __name__ == "__main__":
    import os
    from urllib.request import urlopen

    print("### Job class smoke test ###")

    root_dir = os.path.realpath(os.path.join(os.path.dirname(os.path.realpath(__file__)), '..'))
    jobinfo = {
        'triggers': ['a/1', 'b/2.1', 'c/3.2.1'],
        'ppas': ['ppa:me/myppa']
    }
    job_1 = Job(
        number=0,
        submit_time='time',
        source_package='my-package',
        series='kinetic',
        arch='amd64',
        triggers=jobinfo.get('triggers', None),
        ppas=jobinfo.get('ppas', None)
    )
    print(job_1)
    print(f"triggers:     {job_1.triggers}")
    print(f"ppas:         {job_1.ppas}")
    print(f"request_url:  {job_1.request_url}")
    print()

    ppa = "bryce/dovecot-merge-v1e2.3.19.1adfsg1-2"

    print("running:")
    response = urlopen(f"file://{root_dir}/tests/data/running-20220822.json")
    for job in get_running(response, 'kinetic', ppa):
        print(job)
    print()

    print("waiting:")
    response = urlopen(f"file://{root_dir}/tests/data/queues-20220822.json")
    for job in get_waiting(response, 'kinetic', ppa):
        print(job)
