"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(options) {
        super({
            ...options,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...options?.venvOptions,
            },
            pip: true,
            poetry: false,
            pytest: false,
            setuptools: true,
        });
        if (options.specFile) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib"].forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientLanguages, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in __init__.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Set up pytest manually since the default pytest generates tests for sample code which doesn't exist
        const pytestVersion = options.pytestOptions?.version || "6.2.1";
        this.addDevDependency(`pytest@${pytestVersion}`);
        this.testTask.exec("pytest");
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.3.1" };
//# sourceMappingURL=data:application/json;base64,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