"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiSpecProject = void 0;
const path = require("path");
const fs_extra_1 = require("fs-extra");
const lodash_1 = require("lodash");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const util_1 = require("projen/lib/util");
const parsed_spec_1 = require("./components/parsed-spec");
// initialize logger
const logger = log4js_1.getLogger();
/**
 * Project containing the OpenAPI spec, and a parsed spec for use by the CDK construct
 */
class OpenApiSpecProject extends projen_1.Project {
    constructor(options) {
        super(options);
        this.specChanged = true;
        // Store whether we've synthesized the project
        this.synthed = false;
        logger.trace("OpenApiSpecProject constructor");
        // HACK: remove all components but the ones we are registering - removes .gitignore, tasks, etc since these are
        // unused and a distraction for end-users!
        // @ts-ignore
        this._components = [];
        this.specFileName = options.specFileName ?? "spec.yaml";
        this.parsedSpecFileName = options.parsedSpecFileName ?? ".parsed-spec.json";
        if (!this.parsedSpecFileName.endsWith(".json")) {
            throw new Error("Parsed spec file must end with .json");
        }
        this.specPath = path.join(this.outdir, this.specFileName);
        this.parsedSpecPath = path.join(this.outdir, this.parsedSpecFileName);
        logger.debug(`specPath = "${this.specPath}"`);
        logger.debug(`parsedSpecPath = "${this.parsedSpecPath}"`);
        // Create a sample OpenAPI spec yaml if not defined
        new projen_1.SampleFile(this, this.specFileName, {
            sourcePath: path.join(__dirname, "..", "..", "..", "samples", "spec.yaml"),
        });
        // Check if there is already a .parsed-spec.json present
        const existingParsedSpecJson = util_1.tryReadFileSync(this.parsedSpecPath);
        if (existingParsedSpecJson != null) {
            // generate a new temporary parsed-spec
            const tmpParseSpecPath = `${this.parsedSpecPath}.tmp`;
            logger.trace(`Generating temp spec at "${tmpParseSpecPath}"`);
            parsed_spec_1.ParsedSpec.parse(this.specPath, tmpParseSpecPath);
            // load it
            const newParsedSpecJson = util_1.tryReadFileSync(tmpParseSpecPath);
            if (newParsedSpecJson != null) {
                const prevParseSpec = JSON.parse(existingParsedSpecJson);
                const newParsedSpec = JSON.parse(newParsedSpecJson);
                // check if spec changed and keep it
                logger.trace("Comparing previous and newly generated specs.");
                this.specChanged = !lodash_1.isEqual(prevParseSpec, newParsedSpec);
                logger.debug(`Spec Changed :: ${this.specChanged}`);
                // remove tmp parsed-spec file
                logger.trace("Removing temp spec file.");
                fs_extra_1.unlinkSync(tmpParseSpecPath);
            }
        }
        else {
            logger.debug(`No parsedSpec file found at "${this.parsedSpecPath}".`);
        }
        // Parse the spec to produce a consolidated, bundled spec which can be read by cdk constructs or other tooling
        new parsed_spec_1.ParsedSpec(this, {
            specPath: this.specPath,
            outputPath: this.parsedSpecPath,
        });
    }
    /**
     * @inheritDoc
     */
    synth() {
        // Save some time by only synthesizing once. We synthesize this project early so that it's available for the parent
        // project's install phase (pre-synth). Projen will call this method again at the usual time to synthesize this,
        // project, at which point we're already done so can skip.
        if (this.synthed) {
            logger.trace("OpenApiSpecProject already synthed. Skipping...");
            return;
        }
        logger.trace("OpenApiSpecProject synth");
        super.synth();
        this.synthed = true;
    }
}
exports.OpenApiSpecProject = OpenApiSpecProject;
//# sourceMappingURL=data:application/json;base64,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