"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayTsProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const path = require("path");
const log4js_1 = require("log4js");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const util_1 = require("projen/lib/util");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const typescript_2 = require("./samples/typescript");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const logger = log4js_1.getLogger();
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "@aws-prototyping-sdk/open-api-gateway";
/**
 * Synthesizes a Typescript Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-ts
 */
class OpenApiGatewayTsProject extends typescript_1.TypeScriptProject {
    constructor(projectOptions) {
        super({
            ...projectOptions,
            sampleCode: false,
            tsconfig: {
                ...projectOptions.tsconfig,
                compilerOptions: {
                    lib: ["dom", "es2019"],
                    ...projectOptions.tsconfig?.compilerOptions,
                },
            },
        });
        const options = this.preConstruct(projectOptions);
        if (options.specFile && !path.isAbsolute(options.specFile)) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        logger.debug(`specDir = "${this.specDir}"`);
        logger.debug(`specFileName = "${this.specFileName}"`);
        logger.debug(`generatedCodeDir = "${this.generatedCodeDir}"`);
        logger.debug(`forceGenerateCodeAndDocs = ${this.forceGenerateCodeAndDocs}`);
        logger.debug(`apiSrcDir = "${this.apiSrcDir}"`);
        // Allow json files to be imported (for importing the parsed spec)
        this.tsconfig?.addInclude(`${this.srcdir}/**/*.json`);
        // Set to private since this either uses workspaces or has file dependencies
        this.package.addField("private", true);
        // Generated project should have a dependency on this project, in order to run the generation scripts
        this.addDeps(...[
            OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            "constructs",
            "aws-cdk-lib",
            "cdk-nag",
        ].filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME)));
        // Synthesize the openapi spec early since it's used by the generated typescript client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.srcdir, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
            ...(options.specFile && path.isAbsolute(options.specFile)
                ? {
                    overrideSpecPath: options.specFile,
                }
                : {}),
        });
        spec.synth();
        // Delete the lib directory prior to compilation to ensure latest parsed spec json file is copied
        this.preCompileTask.exec(`rm -rf ${this.libdir}`);
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the typescript client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.TYPESCRIPT);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.TYPESCRIPT,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.package.packageName,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: options.defaultReleaseBranch,
                packageManager: options.packageManager,
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedTypescriptClient = this.generatedClients[languages_1.ClientLanguage.TYPESCRIPT];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedTypescriptClient.synth();
        const typescriptCodeGenDir = path.relative(this.outdir, this.generatedTypescriptClient.outdir);
        if (this.hasParent) {
            // When we have a parent project, we can add a dependency on the generated typescript client since it's part of
            // the monorepo. Since this project will be synthesized before the parent monorepo, the generated typescript
            // client won't be visible for the first install in this project's post synthesize step, so we use a local
            // workspace as a workaround.
            if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
                this.pnpmWorkspace = new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                    readonly: true,
                    obj: {
                        packages: [typescriptCodeGenDir],
                    },
                });
            }
            else {
                this.package.addField("workspaces", {
                    packages: [typescriptCodeGenDir],
                });
            }
            // Add the dependency
            this.addDeps(this.generatedTypescriptClient.package.packageName);
            // Since the generated typescript client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
        }
        else {
            // Add a file dependency on the generated typescript client
            this.addDeps(`${this.generatedTypescriptClient.package.packageName}@file:./${typescriptCodeGenDir}`);
            // Linting runs post synthesize before the typescript client has been built
            this.eslint?.addRules({
                "import/no-unresolved": [
                    2,
                    { ignore: [this.generatedTypescriptClient.package.packageName] },
                ],
            });
        }
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            typescriptClientPackageName: this.generatedTypescriptClient.package.packageName,
            sampleCode: options.sampleCode,
            apiSrcDir: path.join(this.srcdir, this.apiSrcDir),
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
        };
        new projen_1.SampleFile(this, path.join(this.srcdir, "index.ts"), {
            contents: `export * from "./${this.apiSrcDir}";`,
        });
        new projen_1.SampleDir(this, path.join(this.srcdir, this.apiSrcDir), {
            files: typescript_2.getTypescriptSampleSource(sampleOptions),
        });
        new projen_1.SampleDir(this, this.testdir, {
            files: typescript_2.getTypescriptSampleTests(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * This method provides inheritors a chance to synthesize extra resources prior to those created by this project.
     * Return any options you wish to change, other than typescript project options.
     */
    preConstruct(options) {
        return options;
    }
    /**
     * @inheritDoc
     */
    preSynthesize() {
        super.preSynthesize();
    }
    /**
     * @inheritDoc
     */
    postSynthesize() {
        // When no parent is passed, link the generated client as a prebuild step to ensure the latest built generated
        // client is reflected in this package's node modules.
        // Note that it's up to the user to manage building the generated client first.
        !this.hasParent && this.executeLinkNativeClientCommands();
        super.postSynthesize();
    }
    /**
     * Executes commands to link the native (ie typescript) client such that updates to the generated client are reflected
     * in this project's node_modules (when not managed by a monorepo)
     * @private
     */
    executeLinkNativeClientCommands() {
        switch (this.package.packageManager) {
            case javascript_1.NodePackageManager.NPM:
            case javascript_1.NodePackageManager.YARN:
                util_1.exec(`${this.package.packageManager} link`, {
                    cwd: this.generatedTypescriptClient.outdir,
                });
                util_1.exec(`${this.package.packageManager} link ${this.generatedTypescriptClient.package.packageName}`, {
                    cwd: this.outdir,
                });
                break;
            case javascript_1.NodePackageManager.PNPM:
                util_1.exec(`${this.package.packageManager} link ./${path.relative(this.outdir, this.generatedTypescriptClient.outdir)}`, {
                    cwd: this.outdir,
                });
                break;
            default:
                throw new Error(`Unsupported package manager ${this.package.packageManager}`);
        }
    }
}
exports.OpenApiGatewayTsProject = OpenApiGatewayTsProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayTsProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayTsProject", version: "0.12.34" };
//# sourceMappingURL=data:application/json;base64,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